#pragma once

#include "sync_message_op.h"
#include <xeno/operations/environment.h>
#include <xeno/xeno_settings.h>

#include <yplatform/yield.h>

namespace xeno {

struct turbo_sync_op
{
    using msg_info_type = mailbox::msg_info_type;
    using yield_ctx = yplatform::yield_context<turbo_sync_op>;

    turbo_sync_op(const mailbox::folder& folder) : folder(folder)
    {
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::message_vector_ptr messages = {})
    {
        try
        {
            reenter(ctx)
            {
                ENV_LOG(env, info) << "run turbo_sync for folder " << folder.path.to_string();
                state = env.cache_mailbox->sync_newest_state();

                if (folder.count <= env.sync_settings->newest_count)
                {
                    ENV_LOG(env, info) << "no messages for turbo_sync";
                    yield break;
                }

                if (messages_to_download()->empty())
                {
                    calc_range_to_sync(env.sync_settings);
                    ENV_LOG(env, info) << "getting message list from external mailbox";
                    yield env.ext_mailbox->get_messages_info_by_num(
                        folder.path, top, bottom, wrap(env, ctx));
                    if (ec)
                    {
                        if (ec != errc::imap_not_connected)
                        {
                            ENV_LOG(env, error)
                                << "ignoring error during turbo_sync: " << ec.message();
                            ec = {};
                        }
                        yield break;
                    }
                    *messages_to_download() = *messages;
                    if (messages_to_download()->empty())
                    {
                        yield break;
                    }
                }

                ENV_LOG(env, info) << "getting message list from local mailbox";
                yield env.loc_mailbox->get_messages_info_by_id(
                    folder.fid,
                    *mailbox::message_helpers::to_imap_id_vector(messages_to_download()),
                    msg_info_type::without_flags,
                    wrap(env, ctx));
                if (ec) yield break;
                remove_downloaded(messages);
                if (messages_to_download()->empty())
                {
                    yield break;
                }

                std::sort(messages_to_download()->rbegin(), messages_to_download()->rend());
                for (imsg = messages_to_download()->begin(); imsg != messages_to_download()->end();
                     ++imsg)
                {
                    yield spawn<sync_message_op>(
                        wrap(env, ctx),
                        folder.path,
                        imsg->id,
                        "turbo_sync",
                        *imsg,
                        env.cache_mailbox->account().last_sync_ts < imsg->date ?
                            mailbox::notification_type::normal :
                            mailbox::notification_type::disabled);
                    if (ec == code::message_saved_with_error)
                    {
                        ec = code::ok;
                    }
                    if (ec) yield break;
                }
                remove_downloaded(messages_to_download());
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during turbo_sync: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const mailbox::message& /*msg*/)
    {
        (*this)(ctx, env, ec);
    }

    mailbox::message_vector_ptr messages_to_download()
    {
        return state->turbo_sync.folders[folder.path].messages_to_download;
    }

    void calc_range_to_sync(synchronization_settings_ptr settings)
    {
        auto it = settings->turbo_sync.folders.find(folder.type);
        if (it == settings->turbo_sync.folders.end())
        {
            throw std::runtime_error(
                "turbo_sync settings not found, folder=" + folder.path.to_string());
        }
        auto& turbo_sync_settings = it->second;
        if (folder.count <= settings->newest_count)
        {
            throw std::runtime_error(
                "no messages for turbo_sync, folder=" + folder.path.to_string());
        }
        top = folder.count - settings->newest_count;
        bottom = folder.count > turbo_sync_settings.count ?
            folder.count - turbo_sync_settings.count + 1 :
            1;
    }

    void remove_downloaded(mailbox::message_vector_ptr downloaded_messages)
    {
        std::set<mailbox::imap_id_t> downloaded_ids;
        for (auto& msg : *downloaded_messages)
        {
            downloaded_ids.insert(msg.id);
        }
        for (auto it = messages_to_download()->begin(); it != messages_to_download()->end();)
        {
            if (downloaded_ids.count(it->id))
            {
                it = messages_to_download()->erase(it);
            }
            else
            {
                ++it;
            }
        }
    }

    mailbox::folder folder;
    mailbox::sync_newest_state_ptr state;
    mailbox::num_t top = 0;
    mailbox::num_t bottom = 0;
    mailbox::message_vector::iterator imsg;
};

}
#include <yplatform/unyield.h>
