#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct update_and_clear_local_folder_op
{
    using yield_ctx = yplatform::yield_context<update_and_clear_local_folder_op>;

    update_and_clear_local_folder_op(const mailbox::fid_t& fid) : fid(fid)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                ENV_LOG(env, info) << "updating folder in local_mailbox: " << fid;
                folder = env.cache_mailbox->get_folder_by_fid(fid);
                if (!folder)
                {
                    ec = code::folder_not_found;
                    yield break;
                }

                parent_fid = env.cache_mailbox->get_parent_fid_by_child_path(folder->path);
                yield env.loc_mailbox->update_folder(
                    *folder, parent_fid, wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                yield env.loc_mailbox->clear_folder(fid, wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                ENV_LOG(env, info) << "clear folder from local_mailbox: " << fid;
                env.cache_mailbox->set_folder_status_by_path(
                    folder->path, mailbox::folder::status_t::ok);
                env.cache_mailbox->clear_folder(folder->path);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "update and clear local folder op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    mailbox::fid_t fid;
    mailbox::fid_t_opt parent_fid;
    mailbox::folder_opt folder;
};

}
#include <yplatform/unyield.h>
