#pragma once

#include "get_next_sync_oldest_chunk_op.h"
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct update_oldest_messages_cache_op
{
    using msg_info_type = mailbox::msg_info_type;
    using yield_ctx = yplatform::yield_context<update_oldest_messages_cache_op>;

    update_oldest_messages_cache_op(const mailbox::folder& folder) : folder(folder)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {}, mailbox::message_vector_ptr chunk = {})
    {
        try
        {
            reenter(ctx)
            {
                if (!folder.count) yield break;
                env.cache_mailbox->update_oldest_processed_range(
                    folder.path, folder.downloaded_range);

                while (need_more_messages(env.sync_settings->oldest_cache_size))
                {
                    yield spawn<get_next_sync_oldest_chunk_op>(wrap(env, ctx), folder);
                    if (ec) yield break;

                    if (chunk->size() < env.sync_settings->oldest_cache_size)
                    {
                        has_more_messages = false;
                    }

                    messages->insert(messages->end(), chunk->begin(), chunk->end());
                    std::sort(messages->begin(), messages->end());

                    if (messages->empty())
                    {
                        yield break;
                    }

                    ENV_LOG(env, info) << "getting message list from local mailbox";
                    yield env.loc_mailbox->get_messages_info_by_id(
                        folder.fid,
                        *mailbox::message_helpers::to_imap_id_vector(messages),
                        msg_info_type::without_flags,
                        wrap(env, ctx));
                    if (ec) yield break;

                    for (auto& loc_msg : *chunk)
                    {
                        auto msg_it = std::find_if(
                            messages->begin(),
                            messages->end(),
                            [&loc_msg](const mailbox::message& msg) {
                                return loc_msg.id == msg.id;
                            });

                        if (msg_it != messages->end())
                        {
                            messages->erase(msg_it);
                        }
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during update_oldest_messages_cache_op: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            if (!ec)
            {
                env.cache_mailbox->update_oldest_messages_for_sync(folder.path, messages);
            }
            env(ec);
        }
    }

    bool need_more_messages(uint32_t size_limit)
    {
        return messages->size() < size_limit && has_more_messages;
    }

    mailbox::folder folder;
    bool has_more_messages = true;
    mailbox::message_vector_ptr messages = std::make_shared<mailbox::message_vector>();
};

}

#include <yplatform/unyield.h>
