#pragma once

#include <common/errors.h>
#include <mailbox/data_types/redownload_messages_state.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <memory>

namespace xeno {

struct update_redownload_messages_cache_op
{
    using yield_ctx = yplatform::yield_context<update_redownload_messages_cache_op>;

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::message_vector_ptr messages = {})
    {
        try
        {
            reenter(ctx)
            {
                state = env.cache_mailbox->redownload_messages_state();
                state->messages = std::make_shared<mailbox::message_vector>();
                yield env.loc_mailbox->get_not_downloaded_messages(
                    env.sync_settings->redownload_messages_cache_size, wrap(env, ctx));
                if (ec) yield break;

                local_messages = messages;
                std::sort(
                    local_messages->begin(),
                    local_messages->end(),
                    [](const mailbox::message& lhs, const mailbox::message& rhs) {
                        return lhs.errors_count > rhs.errors_count;
                    });

                for (message_it = local_messages->begin(); message_it != local_messages->end();
                     message_it++)
                {
                    path = env.cache_mailbox->get_path_by_fid(message_it->fid);
                    yield env.ext_mailbox->get_messages_info_by_id(
                        path, mailbox::imap_range(message_it->id, message_it->id), wrap(env, ctx));
                    if (ec)
                    {
                        yield break;
                    }
                    if (messages->empty())
                    {
                        yield env.loc_mailbox->delete_mailish_entry(
                            message_it->fid,
                            mailbox::imap_id_vector{ message_it->id },
                            wrap(env, ctx));
                        if (ec) yield break;
                    }
                    else
                    {
                        message_it->flags = messages->begin()->flags;
                        message_it->date = messages->begin()->date;
                        message_it->size = messages->begin()->size;
                        state->messages->push_back(*message_it);
                    }
                }

                state->has_new_failures = false;
                state->has_another_messages_in_local_mb =
                    (local_messages->size() == env.sync_settings->redownload_messages_cache_size);
                if (state->messages->size())
                {
                    state->max_error_count_on_last_cache_update =
                        state->messages->front().errors_count;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during update_redownload_messages_cache: "
                                << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    mailbox::message_vector::iterator message_it;
    mailbox::path_t path;
    mailbox::message_vector_ptr local_messages;

    mailbox::redownload_messages_state_ptr state;
};

}
#include <yplatform/unyield.h>
