#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class clear_folder_op : public boost::asio::coroutine
{
public:
    clear_folder_op(const mailbox::fid_t& fid) : fid_(fid)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "clear folder started: fid=" << fid_;
                path_ = env.cache_mailbox->get_path_by_fid(fid_);

                ENV_LOG(env, info)
                    << "clearing folder in external mailbox: path=" << path_.to_string();
                yield env.ext_mailbox->clear_folder(path_, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                ENV_LOG(env, info) << "clearing folder in local mailbox: fid=" << fid_;
                yield env.loc_mailbox->clear_folder(fid_, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                env.cache_mailbox->clear_folder(path_);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "clear_folder_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

private:
    mailbox::fid_t fid_;
    mailbox::path_t path_;
};

}

#include <boost/asio/unyield.hpp>
