#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <mailbox/data_types/store_request.h>
#include <xeno/operations/user/save_draft_op.h>
#include <xeno/operations/environment.h>

#include "../sync/sync_message_op.h"

#include <yplatform/util/split.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

namespace mb = mailbox;

class compose_and_save_draft_op : public boost::asio::coroutine
{
    using store_request_ptr = store_request_ptr;
    using imap_id_uidvalidity_pair = mailbox::imap_id_uidvalidity_pair;

public:
    compose_and_save_draft_op(
        const std::string& user_ticket,
        store_request_ptr request,
        const std::string& attachments_url)
        : user_ticket(user_ticket), request(request), attachments_url(attachments_url)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "store message started";
                uid = env.cache_mailbox->account().uid;

                ENV_LOG(env, info) << "composing draft";
                yield env.loc_mailbox->compose_draft(
                    uid, user_ticket, request, wrap(env, *this, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                store_fid = parse_store_fid();
                if (store_fid)
                {
                    draft_folder = env.cache_mailbox->get_folder_by_fid(*store_fid);
                }
                else
                {
                    draft_folder =
                        env.cache_mailbox->get_folder_by_type(mb::folder::type_t::drafts);
                }
                if (!draft_folder)
                {
                    ENV_LOG(env, error) << "compose and save draft op error: cannot find "
                                        << (store_fid ? *store_fid : "drafts")
                                        << (store_fid ? " fid" : " folder") << " in cache mailbox";
                    ec = code::folder_not_found;
                    yield break;
                }

                yield spawn<save_draft_op>(
                    wrap(env, *this, uninterruptible),
                    draft_folder->fid,
                    (*compose_draft_resp)["text"].asString());
                if (ec)
                {
                    yield break;
                }
                parse_attachments();
                if (attachment_with_download_url->size())
                {
                    ENV_LOG(env, info) << "getting attachments sids";
                    yield get_attachments_sids(env);
                    if (ec)
                    {
                        yield break;
                    }
                }
                make_response();
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "compose and save draft op error: exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec, std::move(*result));
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, json::value response)
    {
        compose_draft_resp = std::make_shared<json::value>(std::move(response));
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error ec, mailbox::mid_t mid)
    {
        new_draft_mid = mid;
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error ec, std::vector<std::string> sids)
    {
        if (ec) return (*this)(env, ec);

        if (attachment_with_download_url->size() != sids.size())
        {
            ENV_LOG(env, error) << "compose and save draft op error: get attachments sids error: "
                                << "response sids size=" << sids.size()
                                << ", but attachments size=" << *attachment_with_download_url;
            ec = code::cannot_get_attachments_sids;
            return (*this)(env, ec);
        }

        try
        {
            auto attachment_it = attachment_with_download_url->begin();
            auto attachment_end = attachment_with_download_url->end();
            auto sid_it = sids.begin();
            for (; attachment_it != attachment_end; ++attachment_it, ++sid_it)
            {
                (*attachment_it)["download_url"] =
                    make_download_url((*attachment_it)["name_uri_encoded"].asString(), *sid_it);
                (*attachment_it).removeMember("name_uri_encoded");
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error)
                << "compose and save draft op error: get attachments sids exception: " << e.what();
            ec = code::cannot_get_attachments_sids;
        }

        (*this)(env, ec);
    }

    template <typename Env>
    void get_attachments_sids(Env&& env)
    {
        std::vector<std::string> hids;
        hids.reserve(attachment_with_download_url->size());
        for (auto& attachment : *attachment_with_download_url)
        {
            hids.emplace_back(attachment["hid"].asString());
        }

        env.loc_mailbox->get_attachments_sids(
            uid, user_ticket, new_draft_mid, hids, wrap(env, *this, uninterruptible));
    }

    void make_response()
    {
        json::value attachments(Json::arrayValue);
        for (auto& attachment : *attachment_without_download_url)
        {
            attachments.append(attachment);
        }
        for (auto& attachment : *attachment_with_download_url)
        {
            attachments.append(attachment);
        }

        (*result)["attachments"]["attachment"] = attachments;
        (*result)["fid"] = draft_folder->fid;
        (*result)["mid"] = std::to_string(new_draft_mid);
    }

    mb::fid_t_opt parse_store_fid()
    {
        auto store_fid = request->get_value("store_fid");
        if (store_fid.size())
        {
            return store_fid;
        }
        return {};
    }

    void parse_attachments()
    {
        auto& attachments = (*compose_draft_resp)["attachments"];
        for (auto& attachment : attachments)
        {
            json::value result;
            result["hid"] = attachment["hid"];
            if (!attachment["old_hid"].isNull())
            {
                result["old_hid"] = attachment["old_hid"];
            }
            if (attachment["name"] == "narod_attachment_links.html")
            {
                result["narod"] = true;
                attachment_without_download_url->append(result);
            }
            else
            {
                result["narod"] = false;
                result["display_name"] = attachment["name"];
                result["class"] = attachment["clas"];
                result["size"] = attachment["length"];
                result["mime_type"] = parse_mime_type(attachment);
                result["is_inline"] = (!attachment["is_inline"].isNull());
                result["name_uri_encoded"] = attachment["name_uri_encoded"];

                if (!attachment["preview-supported"].isNull())
                {
                    result["preview-supported"] = attachment["preview-supported"];
                }

                if (!attachment["cid"].isNull())
                {
                    result["content_id"] = attachment["cid"];
                }
                attachment_with_download_url->append(result);
            }
        }
    }

    std::string parse_mime_type(const json::value& attachment)
    {
        std::stringstream result;
        auto subtype = yplatform::util::split(attachment["subtype"].asString(), ";");
        result << attachment["type"].asString() << "/" << subtype.at(0);
        return result.str();
    }

    std::string make_download_url(const std::string& attachment_name, const std::string& sid)
    {
        std::stringstream result;
        result << attachments_url << "/message_part_real"
               << "/" << attachment_name << "?sid=" << sid
               << "&no_disposition=y&thumb=y&exif_rotate=y";
        return result.str();
    }

private:
    uid_t uid{ 0 };
    const std::string user_ticket;
    store_request_ptr request;
    std::string attachments_url;
    json::value_ptr compose_draft_resp{ std::make_shared<json::value>() };
    mb::fid_t_opt store_fid;

    mb::folder_opt draft_folder;
    mb::mid_t new_draft_mid{ 0 };
    json::value_ptr attachment_without_download_url{ std::make_shared<json::value>() };
    json::value_ptr attachment_with_download_url{ std::make_shared<json::value>() };

    json::value_ptr result{ std::make_shared<json::value>() };
    ;
};

}

#include <boost/asio/unyield.hpp>
