#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <mailbox/data_types/store_request.h>
#include <xeno/operations/user/send_op.h>
#include <xeno/operations/sync/sync_message_op.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>
#include <boost/algorithm/string/find.hpp>

namespace xeno::user {

namespace mb = mailbox;

class compose_and_send_op : public boost::asio::coroutine
{
    using send_request_ptr = send_request_ptr;

public:
    compose_and_send_op(const std::string& user_ticket, send_request_ptr request)
        : user_ticket(user_ticket)
        , request(request)
        , operation_id(request->get_value("operation_id"))
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "send message started";
                uid = env.cache_mailbox->account().uid;
                yield env.loc_mailbox->get_send_operation_result(
                    operation_id, wrap(env, *this, uninterruptible));
                if (saved_operation_result)
                {
                    ENV_LOG(env, info)
                        << "send operaion was already completed, operation_id=" << operation_id
                        << ", mid=" << *saved_operation_result;
                    yield break;
                }

                email = request->get_value("from_mailbox");
                if (email.empty())
                {
                    email = env.cache_mailbox->account().email;
                }
                ENV_LOG(env, info) << "composing message";
                yield env.loc_mailbox->compose_message(
                    uid, user_ticket, request, wrap(env, *this, uninterruptible));
                if (ec)
                {
                    yield break;
                }
                parse_rcpts();
                yield send(env);
                if (ec) yield break;
                ENV_LOG(env, info) << "send operation completed"
                                   << (operation_id.size() ? ", operation_id=" + operation_id : "")
                                   << ", mid=" << saved_mid;

                yield env.loc_mailbox->save_send_operation_result(
                    operation_id, std::to_string(saved_mid), wrap(env, *this, uninterruptible));
                if (ec)
                {
                    // just ignore any error, message was sent successfully
                    // log has been written in local mailbox
                    ec = code::ok;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "compose and send op error: exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec, is_duplicated_send());
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, json::value response)
    {
        compose_send_resp = std::make_shared<json::value>(response);
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error ec, mailbox::mid_t mid)
    {
        saved_mid = mid;
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error /*ec*/, string_opt cache_res)
    {
        // just ignore any error (log has been written in local mailbox)
        saved_operation_result = cache_res;
        (*this)(std::forward<Env>(env), code::ok);
    }

private:
    template <typename Env>
    void send(Env&& env)
    {
        auto client_ip = request->get_value("x-real-ip");
        auto request_id = request->get_value("x-request-id");
        if (client_ip.empty() || request_id.empty())
        {
            ENV_LOG(env, error) << "compose and send op error: header not found: "
                                << (client_ip.empty() ? "x-real-ip" : "x-request-id");
            return (*this)(std::forward<Env>(env), code::bad_request);
        }
        spawn<send_op>(
            wrap(env, *this, uninterruptible),
            email,
            rcpts,
            client_ip,
            request_id,
            (*compose_send_resp)["text"].asString());
    }

    void parse_rcpts()
    {
        parse_rcpt("to", rcpts);
        parse_rcpt("cc", rcpts);
        parse_rcpt("bcc", rcpts);
    }

    void parse_rcpt(const std::string& key, std::vector<std::string>& target)
    {
        auto rcpt = (*compose_send_resp)[key];
        if (rcpt.isArray())
        {
            for (auto& it : rcpt)
            {
                target.push_back(json::to_string(it));
            }
        }
    }

    bool is_mail_for_himself(const std::string& email)
    {
        return std::count_if(rcpts.begin(), rcpts.end(), [email](const std::string& rcpt) {
            return boost::ifind_first(rcpt, email);
        });
    }

    bool is_duplicated_send()
    {
        return saved_operation_result.has_value();
    }

    uid_t uid{ 0 };
    std::string user_ticket;
    std::string email;
    send_request_ptr request;
    std::string operation_id;
    std::optional<std::string> saved_operation_result;
    mailbox::mid_t saved_mid;
    json::value_ptr compose_send_resp{ std::make_shared<json::value>() };
    std::vector<std::string> rcpts;
    mb::folder_opt sent_folder;
    error send_error;
};

}

#include <boost/asio/unyield.hpp>
