#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class create_folder_op : public boost::asio::coroutine
{
public:
    create_folder_op(
        const std::string& name,
        const mailbox::fid_t& parent_fid,
        const std::string& symbol)
        : name_(name), parent_fid_(parent_fid), symbol_(symbol)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info)
                    << "create folder started: name=" << name_ << ", parent_fid=" << parent_fid_;
                full_path = env.cache_mailbox->get_path_by_fid(parent_fid_).make_child_path(name_);

                ENV_LOG(env, info)
                    << "creating folder in external mailbox: path=" << full_path.to_string();
                yield env.ext_mailbox->create_folder(full_path, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                ENV_LOG(env, info)
                    << "creating folder in local mailbox: path=" << full_path.to_string()
                    << ", parent_fid=" << parent_fid_;
                yield env.loc_mailbox->create_folder(
                    mailbox::folder(full_path),
                    parent_fid_,
                    symbol_,
                    wrap(env, *this, uninterruptible));
                if (ec) yield break;

                env.cache_mailbox->update_folders_from_local(created_folder_);
                if (created_folder_->size())
                {
                    res_fid_ = created_folder_->at(0).fid;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "create_folder_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec, res_fid_);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, mailbox::folder_vector_ptr res)
    {
        created_folder_ = res;
        (*this)(std::forward<Env>(env), ec);
    }

private:
    std::string name_;
    mailbox::fid_t parent_fid_;
    std::string symbol_;
    mailbox::fid_t res_fid_;

    mailbox::path_t full_path;
    mailbox::folder_vector_ptr created_folder_;
};

}

#include <boost/asio/unyield.hpp>
