#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include "../sync/delete_messages_from_folder_op.h"

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class delete_folder_op : public boost::asio::coroutine
{
public:
    delete_folder_op(const mailbox::fid_t& fid) : fid_(fid)
    {
    }

    template <typename Env>
    void operator()(
        Env&& env,
        error ec = {},
        mailbox::imap_id_transform_map_ptr /*imap_id_transform_map*/ = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "delete folder started: fid=" << fid_;
                folder_ = env.cache_mailbox->get_folder_by_fid(fid_);

                if (!folder_)
                {
                    ec = code::folder_not_found;
                    yield break;
                }

                ENV_LOG(env, info)
                    << "deleting synced messages: path=" << folder_->path.to_string();
                do
                {
                    yield env.loc_mailbox->get_messages_info_top(
                        fid_,
                        env.sync_settings->delete_messages_chunk_size,
                        wrap(env, *this, uninterruptible));
                    if (ec) yield break;
                    if (messages_->empty()) break;
                    yield spawn<delete_messages_from_folder_op>(
                        wrap(env, *this, uninterruptible), *folder_, messages_);
                    if (ec) yield break;
                } while (messages_->size() == env.sync_settings->delete_messages_chunk_size);

                trash_folder_ =
                    env.cache_mailbox->get_folder_by_type(mailbox::folder::type_t::trash);
                if (trash_folder_)
                {
                    ENV_LOG(env, info) << "moving messages to trash in external mailbox: path="
                                       << folder_->path.to_string();
                    yield env.ext_mailbox->move_all_messages(
                        folder_->path, trash_folder_->path, wrap(env, *this, uninterruptible));
                    if (ec) yield break;
                }
                else
                {
                    ENV_LOG(env, info) << "clearing folder in external mailbox: path="
                                       << folder_->path.to_string();
                    yield env.ext_mailbox->clear_folder(
                        folder_->path, wrap(env, *this, uninterruptible));
                    if (ec) yield break;
                }

                ENV_LOG(env, info)
                    << "deleting folder in external mailbox: path=" << folder_->path.to_string();
                yield env.ext_mailbox->delete_folder(
                    folder_->path, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                env.cache_mailbox->set_folder_status_by_path(
                    folder_->path, mailbox::folder::status_t::to_clear_and_delete);

                ENV_LOG(env, info) << "clearing folder in local mailbox: fid=" << fid_;
                yield env.loc_mailbox->clear_folder(fid_, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                ENV_LOG(env, info) << "deleting folder in local mailbox: fid=" << fid_;
                yield env.loc_mailbox->delete_folder(fid_, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                env.cache_mailbox->delete_folder_by_fid(fid_);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "delete_folder_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, mailbox::message_vector_ptr messages)
    {
        if (!ec)
        {
            messages_ = messages;
        }
        (*this)(env, ec);
    }

private:
    mailbox::fid_t fid_;
    mailbox::folder_opt folder_;
    mailbox::folder_opt trash_folder_;
    mailbox::message_vector_ptr messages_;
};

}

#include <boost/asio/unyield.hpp>
