#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>
#include <xeno/operations/user/delete_folder_op.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class delete_folder_with_subfolders_op : public boost::asio::coroutine
{
public:
    delete_folder_with_subfolders_op(const mailbox::fid_t& fid) : fid_(fid)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "delete folder with subfolders started: fid=" << fid_;
                subfolders_ = env.cache_mailbox->get_subfolders(fid_);
                if (!check_subfolders_fids(env))
                {
                    ec = code::folder_bad_status;
                    yield break;
                }
                for (ifolder_ = subfolders_->rbegin(); ifolder_ != subfolders_->rend(); ++ifolder_)
                {
                    ENV_LOG(env, info) << "deleting subfolder fid=" << ifolder_->fid;
                    yield spawn<delete_folder_op>(wrap(env, *this, uninterruptible), ifolder_->fid);
                    if (ec) yield break;
                }
                yield spawn<delete_folder_op>(wrap(env, *this, uninterruptible), fid_);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "delete_folder_with_subfolders_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    bool check_subfolders_fids(Env&& env)
    {
        for (auto& folder : *subfolders_)
        {
            if (folder.fid.empty())
            {
                ENV_LOG(env, info) << "delete folder with subfolders error: subfolder with path= "
                                   << folder.path.to_string() << " hasn't fid";
                return false;
            }
        }
        return true;
    }

private:
    mailbox::fid_t fid_;
    mailbox::path_t path_;
    mailbox::folder_vector_ptr subfolders_;
    mailbox::folder_vector::reverse_iterator ifolder_;
};

}

#include <boost/asio/unyield.hpp>
