#pragma once

#include "../sync/move_messages_op.h"

#include <common/errors.h>
#include <common/split.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class move_messages_op : public boost::asio::coroutine
{
public:
    move_messages_op(
        const mailbox::mid_vector& mids,
        const mailbox::tid_vector& tids,
        const mailbox::fid_t& dst_fid,
        const mailbox::tab_opt& dst_tab)
        : mids_(std::make_shared<mailbox::mid_vector>(mids))
        , tids_(std::make_shared<mailbox::tid_vector>(tids))
        , dst_fid_(dst_fid)
        , dst_tab_(dst_tab)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "move messages started";

                ENV_LOG(env, info) << "resolving tids to mids";
                yield env.loc_mailbox->get_mids_by_tids(*tids_, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                mid_chunks_ = split_for_chunks(*mids_, env.sync_settings->user_op_chunk_size);

                for (chunk_number_ = 0; chunk_number_ != mid_chunks_.size(); ++chunk_number_)
                {
                    ENV_LOG(env, info) << "moving messages at chunk " << chunk_number_ + 1 << " of "
                                       << mid_chunks_.size();
                    yield spawn<::xeno::move_messages_op>(
                        wrap(env, *this, uninterruptible),
                        mid_chunks_[chunk_number_],
                        dst_fid_,
                        dst_tab_);
                    if (ec) yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "move_messages_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, mailbox::mid_vector_ptr res)
    {
        if (!ec)
        {
            mids_->insert(mids_->end(), res->begin(), res->end());
        }
        (*this)(std::forward<Env>(env), ec);
    }

private:
    mailbox::mid_vector_ptr mids_;
    mailbox::tid_vector_ptr tids_;

    mailbox::fid_t dst_fid_;
    mailbox::tab_opt dst_tab_;

    size_t chunk_number_;
    mailbox::mid_chunks mid_chunks_;
};

}

#include <boost/asio/unyield.hpp>
