#pragma once

#include <mailbox/data_types/folder.h>
#include <xeno/operations/sync/sync_message_op.h>
#include <xeno/operations/sync/delete_messages_from_folder_op.h>
#include <xeno/operations/environment.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

namespace mb = mailbox;

class save_draft_op : public boost::asio::coroutine
{
    using store_request_ptr = store_request_ptr;
    using imap_id_uidvalidity_pair = mb::imap_id_uidvalidity_pair;

public:
    save_draft_op(const mb::fid_t& fid, std::string&& body)
        : fid(fid), body(std::make_shared<std::string>(std::move(body)))
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "save draft started";
                folder = env.cache_mailbox->get_folder_by_fid(fid);
                if (!folder)
                {
                    ENV_LOG(env, error)
                        << "save draft op error: " << fid << " fid nod found in cache mailbox";
                    ec = code::folder_not_found;
                    yield break;
                }

                ENV_LOG(env, info) << "appending message to " << folder->path.get_name();
                yield env.ext_mailbox->append(
                    folder->path,
                    std::move(*body),
                    mb::flags_t({ mb::system_flag_t::draft, mb::system_flag_t::seen }, {}),
                    "",
                    wrap(env, *this, uninterruptible));

                if (ec)
                {
                    yield break;
                }

                ENV_LOG(env, info) << "syncing message back from " << folder->path.get_name();
                yield spawn<sync_message_op>(
                    wrap(env, *this, uninterruptible),
                    folder->path,
                    imap_id,
                    "save_draft",
                    mailbox::message_opt(),
                    mailbox::notification_type::disabled,
                    false);
                if (ec)
                {
                    // We must delete draft
                    ENV_LOG(env, info) << "deleting message because of error: " << ec.message();
                    yield spawn<::xeno::delete_messages_from_folder_op>(
                        wrap(env, *this, uninterruptible),
                        *folder,
                        std::make_shared<mailbox::message_vector>(1, mb::message(imap_id)),
                        true);

                    ec = code::cannot_sync_draft;
                    yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "save draft op error: exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec, mid);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, imap_id_uidvalidity_pair append_result)
    {
        // ignore uidvalidity, because we keep uidvalidity per folder and change we resolve in
        // another operations
        // TODO Resolve for imaps without UIDPLUS extension
        imap_id = append_result.first;
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error ec, const mailbox::message& msg)
    {
        this->mid = msg.mid;
        (*this)(std::forward<Env>(env), ec);
    }

private:
    mb::fid_t fid;
    mb::folder_opt folder;
    std::shared_ptr<std::string> body;
    mb::imap_id_t imap_id{ 0 };
    mb::mid_t mid{ 0 };
};

}

#include <boost/asio/unyield.hpp>
