#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include "../sync/sync_message_op.h"
#include "../sync/delete_messages_from_folder_op.h"

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

namespace mb = mailbox;

class send_op : public boost::asio::coroutine
{
    using send_request_ptr = send_request_ptr;
    using imap_id_uidvalidity_pair = mailbox::imap_id_uidvalidity_pair;

public:
    send_op(
        const std::string& email,
        const std::vector<std::string>& rcpts,
        const std::string& ip,
        const std::string& request_id,
        std::string&& body,
        bool notify = false,
        bool save_to_sent = true)
        : email(email)
        , rcpts(rcpts)
        , ip(ip)
        , request_id(request_id)
        , body(std::make_shared<std::string>(std::move(body)))
        , notify(notify)
        , save_to_sent(save_to_sent)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                if (email != env.cache_mailbox->account().email)
                {
                    ENV_LOG(env, error)
                        << "send op error: request_email=" << email
                        << " but validated_email=" << env.cache_mailbox->account().email;
                    ec = code::bad_request;
                    yield break;
                }

                if (rcpts.empty())
                {
                    ENV_LOG(env, error) << "send op error: rcpts are empty";
                    ec = code::bad_request;
                    yield break;
                }

                if (body->empty())
                {
                    ec = code::body_is_empty;
                    yield break;
                }

                yield env.ext_mailbox->get_provider(
                    env.cache_mailbox->account().imap_ep, wrap(env, *this, uninterruptible));
                if (ec)
                {
                    ENV_LOG(env, error) << "send op error: get provider error: " << ec.message();
                    yield break;
                }
                save_to_sent = (provider == "outlook" ? false : save_to_sent);

                ENV_LOG(env, info) << "checking for spam message";
                yield check_spam(env);
                if (ec)
                {
                    yield break;
                }

                sent_folder = env.cache_mailbox->get_folder_by_type(mb::folder::type_t::sent);
                if (save_to_sent && !sent_folder)
                {
                    ENV_LOG(env, info) << "don't append message because sent folder not found";
                    save_to_sent = false;
                }

                append_after_send =
                    provider == "gmail" && is_mail_for_himself(env.cache_mailbox->account().email);
                if (save_to_sent && !append_after_send)
                {
                    yield append(env);

                    if (ec)
                    {
                        yield break;
                    }
                }

                ENV_LOG(env, info) << "sending message";
                yield env.ext_mailbox->send(
                    env.cache_mailbox->account().smtp_login,
                    *env.cache_mailbox->account().auth_data.begin(),
                    env.cache_mailbox->account().smtp_ep,
                    email,
                    rcpts,
                    body,
                    notify,
                    wrap(env, *this, uninterruptible));

                if (ec)
                {
                    if (sent_id)
                    {
                        send_error = ec;
                        ENV_LOG(env, info) << "deleting appended message because " << ec.message();
                        yield spawn<::xeno::delete_messages_from_folder_op>(
                            wrap(env, *this, uninterruptible),
                            *sent_folder,
                            std::make_shared<mb::message_vector>(1, mb::message(sent_id)),
                            true);
                        if (ec)
                        {
                            ENV_LOG(env, error)
                                << "send op error: delete appended message error: " << ec.message();
                        }
                        // because we should return sent error
                        ec = send_error;
                    }
                    yield break;
                }

                if (save_to_sent && append_after_send)
                {
                    yield append(env);

                    if (ec)
                    {
                        yield break;
                    }
                }

                if (!sent_id)
                {
                    yield break;
                }

                ENV_LOG(env, info) << "syncing message back from sent folder";
                yield spawn<sync_message_op>(
                    wrap(env, *this, uninterruptible),
                    sent_folder->path,
                    sent_id,
                    "send",
                    mb::message_opt(),
                    mailbox::notification_type::disabled);

                if (ec)
                {
                    ENV_LOG(env, error)
                        << "send op error: sync appended message error: " << ec.message();
                    // do not respond error to client, message was sent succesfully, it would be
                    // synced later
                    ec = code::ok;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "send op error: exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec, mid);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, imap_id_uidvalidity_pair append_result)
    {
        if (!ec)
        {
            // ignore uidvalidity, because we keep uidvalidity per folder and change we resolve in
            // another operations
            // TODO Resolve for imaps without UIDPLUS extension
            sent_id = append_result.first;
        }
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error ec, const mailbox::message& msg)
    {
        this->mid = msg.mid;
        (*this)(std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(Env&& env, error ec, const std::string provider)
    {
        if (!ec)
        {
            this->provider = provider;
        }
        (*this)(std::forward<Env>(env), ec);
    }

private:
    template <typename Env>
    void check_spam(Env&& env)
    {
        env.loc_mailbox->check_spam(
            env.cache_mailbox->account().uid,
            env.cache_mailbox->account().karma,
            email,
            rcpts,
            ip,
            request_id,
            body,
            wrap(env, *this, uninterruptible));
    }

    template <typename Env>
    void append(Env&& env)
    {
        ENV_LOG(env, info) << "appending message to " << sent_folder->path.to_string() << " folder";
        env.ext_mailbox->append(
            sent_folder->path,
            *body,
            mb::flags_t({ mb::system_flag_t::seen }, {}),
            "",
            wrap(env, *this, uninterruptible));
    }

    bool is_mail_for_himself(const std::string& email)
    {
        return std::count_if(rcpts.begin(), rcpts.end(), [email](const std::string& rcpt) {
            return boost::ifind_first(rcpt, email);
        });
    }

    std::string email;
    std::vector<std::string> rcpts;
    std::string ip;
    std::string request_id;
    std::shared_ptr<std::string> body;
    bool notify;
    bool save_to_sent;
    bool append_after_send;
    std::string provider;
    mb::folder_opt sent_folder;
    mb::imap_id_t sent_id{ 0 };
    mb::mid_t mid{ 0 };
    error send_error;
};

}

#include <boost/asio/unyield.hpp>
