#pragma once

#include <common/errors.h>
#include <common/split.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include "../sync/change_flags_op.h"

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class set_read_flag_op : public boost::asio::coroutine
{
public:
    set_read_flag_op(const mailbox::mid_vector& mids, const mailbox::tid_vector& tids, bool read)
        : mids_(std::make_shared<mailbox::mid_vector>(mids))
        , tids_(std::make_shared<mailbox::tid_vector>(tids))
    {
        if (read)
        {
            add_flags_.system_flags.insert(mailbox::system_flag_t::seen);
        }
        else
        {
            del_flags_.system_flags.insert(mailbox::system_flag_t::seen);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "set read flag started";
                ENV_LOG(env, info) << "resolving tids to mids";
                yield env.loc_mailbox->get_mids_by_tids(*tids_, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                mid_chunks_ = split_for_chunks(*mids_, env.sync_settings->user_op_chunk_size);

                for (chunk_number_ = 0; chunk_number_ != mid_chunks_.size(); ++chunk_number_)
                {
                    ENV_LOG(env, info) << "change flags at message chunk " << chunk_number_ + 1
                                       << " of " << mid_chunks_.size();
                    yield spawn<::xeno::change_flags_op>(
                        wrap(env, *this, uninterruptible),
                        mid_chunks_[chunk_number_],
                        add_flags_,
                        del_flags_);
                    if (ec) yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "set_read_flag_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(Env&& env, error ec, mailbox::mid_vector_ptr res)
    {
        if (!ec)
        {
            mids_->insert(mids_->end(), res->begin(), res->end());
        }
        (*this)(std::forward<Env>(env), ec);
    }

private:
    mailbox::mid_vector_ptr mids_;
    mailbox::tid_vector_ptr tids_;
    mailbox::flags_t add_flags_;
    mailbox::flags_t del_flags_;

    size_t chunk_number_;
    mailbox::mid_chunks mid_chunks_;
};

}

#include <boost/asio/unyield.hpp>
