#pragma once

#include "move_messages_op.h"

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class set_spam_status_op : public boost::asio::coroutine
{
public:
    set_spam_status_op(const mailbox::mid_vector& mids, const mailbox::tid_vector& tids, bool spam)
        : mids_(std::make_shared<mailbox::mid_vector>(mids))
        , tids_(std::make_shared<mailbox::tid_vector>(tids))
    {
        dst_folder_type = spam ? mailbox::folder::type_t::spam : mailbox::folder::type_t::inbox;
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "set spam status started";
                dst_folder = env.cache_mailbox->get_folder_by_type(dst_folder_type);
                if (!dst_folder)
                {
                    ec = code::folder_not_found;
                    yield break;
                }

                yield move_messages_op(*mids_, *tids_, dst_folder->fid, std::nullopt)(
                    wrap(env, *this, uninterruptible));
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "set_spam_status_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

private:
    mailbox::mid_vector_ptr mids_;
    mailbox::tid_vector_ptr tids_;
    mailbox::folder::type_t dst_folder_type;
    mailbox::folder_opt dst_folder;
};

}

#include <boost/asio/unyield.hpp>
