#pragma once

#include <common/errors.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno::user {

class update_folder_op : public boost::asio::coroutine
{
public:
    update_folder_op(
        const mailbox::fid_t& fid,
        const std::string& name,
        const mailbox::fid_t_opt& parent_fid)
        : fid_(fid), parent_fid_(parent_fid), name_(name)
    {
    }

    template <typename Env>
    void operator()(Env&& env, error ec = {})
    {
        try
        {
            reenter(this)
            {
                ENV_LOG(env, info) << "update folder started: fid=" << fid_ << ", name=" << name_;
                folder_ = env.cache_mailbox->get_folder_by_fid(fid_);
                if (!folder_)
                {
                    ec = code::folder_not_found;
                    yield break;
                }
                old_path_ = folder_->path;
                folder_->path = parent_fid_ ? env.cache_mailbox->get_path_by_fid(*parent_fid_) :
                                              old_path_.get_parent_path();

                if (name_.empty())
                {
                    name_ = old_path_.get_name();
                }
                folder_->path = folder_->path.make_child_path(name_);

                ENV_LOG(env, info)
                    << "renaming folder in external mailbox: old_path=" << old_path_.to_string()
                    << ", new_path=" << folder_->path.to_string();
                yield env.ext_mailbox->rename_folder(
                    old_path_, folder_->path, wrap(env, *this, uninterruptible));
                if (ec) yield break;

                ENV_LOG(env, info) << "renaming folder in local mailbox";
                yield env.loc_mailbox->update_folder(
                    *folder_, parent_fid_, wrap(env, *this, uninterruptible));
                if (!ec)
                {
                    env.cache_mailbox->rename_folder(old_path_, folder_->path, parent_fid_);
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "update_folder_op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (is_complete())
        {
            env(ec);
        }
    }

private:
    mailbox::fid_t fid_;
    mailbox::folder_opt folder_;
    mailbox::fid_t_opt parent_fid_;
    std::string name_;

    mailbox::path_t old_path_;
};

}

#include <boost/asio/unyield.hpp>
