#pragma once

#include <xeno/sync_phase.h>

#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/util/tuple_unpack.h>
#include <yplatform/find.h>

#include <tuple>
#include <memory>
#include <utility>
#include <functional>

namespace xeno::rc {

using rate_controller_module = ymod_ratecontroller::rate_controller_module;
using rate_controller_ptr = ymod_ratecontroller::rate_controller_ptr;

template <typename Tuple, std::size_t... Is>
inline auto get_tuple_prefix(Tuple&& tuple, std::index_sequence<Is...>)
{
    return std::make_tuple(std::get<Is>(std::forward<Tuple>(tuple))...);
}

template <std::size_t length, typename Tuple>
inline auto get_tuple_prefix(Tuple&& tuple)
{
    return get_tuple_prefix(std::forward<Tuple>(tuple), std::make_index_sequence<length>());
}

template <typename Handler>
struct completion_wrapper
{
    template <typename... Args>
    void operator()(Args&&... args)
    {
        yplatform::safe_call(on_complete);
        handler(std::forward<Args>(args)...);
    }

    Handler handler;
    std::function<void()> on_complete;
};

class rate_controllers_storage
{
public:
    rate_controllers_storage(const std::string& prefix = {}, const std::string& suffix = {})
    {
        auto module = yplatform::find<rate_controller_module, std::shared_ptr>("rate_controller");
        rate_controllers_["first_sync"] = module->get_controller(prefix + "first_sync" + suffix);
        rate_controllers_["user_op"] = module->get_controller(prefix + "user_op" + suffix);
        rate_controllers_["sync_newest"] = module->get_controller(prefix + "sync_newest" + suffix);
        rate_controllers_["sync_oldest"] = module->get_controller(prefix + "sync_oldest" + suffix);
    }

    template <typename Env>
    rate_controller_ptr get_rate_controller(Env&& env) const
    {
        if (env.cache_mailbox->has_security_lock())
        {
            return rate_controllers_.at("first_sync");
        }
        switch (env.sync_phase)
        {
        case sync_phase::user_op:
            return rate_controllers_.at("user_op");
        case sync_phase::initial:
        case sync_phase::sync_folders:
        case sync_phase::sync_newest:
            return rate_controllers_.at("sync_newest");
        case sync_phase::sync_oldest:
        case sync_phase::sync_oldest_flags_and_deleted:
        case sync_phase::redownload_messages:
        default:
            return rate_controllers_.at("sync_oldest");
        }
    }

    void cancel(yplatform::task_context_ptr ctx)
    {
        // cancel for all phases because we dont know current sync_phase
        for (auto& [name, rc] : rate_controllers_)
        {
            rc->cancel(ctx->uniq_id());
        }
    }

private:
    std::map<std::string, rate_controller_ptr> rate_controllers_;
};

/*
 * call_with_rc receive method and args, extract env from args,
 * wrap it by completion_wrapper and post this method to rate_controller
 */
template <typename... Result, typename Method, typename ArgsTuple>
void call_with_rc(
    context_ptr context,
    const rate_controllers_storage& rc_storage,
    Method method,
    ArgsTuple&& args_tuple)
{
    auto env = std::move(std::get<std::tuple_size<ArgsTuple>::value - 1>(args_tuple));
    auto args_without_env = get_tuple_prefix<std::tuple_size<ArgsTuple>::value - 1>(
        std::forward<ArgsTuple>(args_tuple));

    auto rc = rc_storage.get_rate_controller(env);
    rc->post(
        [method, args_without_env = std::move(args_without_env), env](
            error err, std::function<void()> on_complete) mutable {
            completion_wrapper<decltype(env)> wrapped_env = { env, on_complete };

            if (err)
            {
                wrapped_env(err, Result()...);
            }
            else
            {
                try
                {
                    yplatform::util::call_with_tuple_args(
                        std::mem_fn(method), std::move(args_without_env), wrapped_env);
                }
                catch (const std::exception& e)
                {
                    wrapped_env(error(code::call_with_rc_exception, e.what()), Result()...);
                }
            }
        },
        context->uniq_id(),
        env.deadline);
}

}
