#pragma once

#include "common.h"

#include <mailbox/external/external_mailbox_impl.h>
#include <boost/algorithm/string/replace.hpp>

namespace xeno::rc {

namespace mb = mailbox;

/**
 * rc::ext_mailbox proxy all requests to ext_mailbox through rate_controller
 */
class ext_mailbox : public std::enable_shared_from_this<ext_mailbox>
{
public:
    using mailbox = mb::external::ext_mailbox_impl;

    ext_mailbox(
        context_ptr context,
        std::shared_ptr<mailbox> ext_mb,
        const std::string& imap_server)
        : context_(context)
        , ext_mb_(ext_mb)
        , rc_storage_("external_mailbox." + boost::replace_all_copy(imap_server, ".", "_") + ".")
    {
    }

    void cancel()
    {
        rc_storage_.cancel(context_);
    }

    template <typename... Args>
    void imap_authorize(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::imap_authorize,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_provider(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::get_provider,
            make_args_tuple(std::forward<Args>(args)...));
    }

    std::string get_provider_unsafe()
    {
        return ext_mb_->get_provider_unsafe();
    }

    template <typename... Args>
    void get_folder_vector(Args&&... args)
    {
        call_with_rc<mb::folder_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_folder_vector,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_folder_info(Args&&... args)
    {
        call_with_rc<mb::folder_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_folder_info,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_messages_info_by_id(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_messages_info_by_id,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_messages_info_by_num(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_messages_info_by_num,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_message_body(Args&&... args)
    {
        call_with_rc<mb::string_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_message_body,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_messages(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_messages,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void move_messages(Args&&... args)
    {
        call_with_rc<mb::imap_id_transform_map_ptr>(
            context_,
            rc_storage_,
            &mailbox::move_messages,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void move_all_messages(Args&&... args)
    {
        call_with_rc<mb::imap_id_transform_map_ptr>(
            context_,
            rc_storage_,
            &mailbox::move_all_messages,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void create_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::create_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void rename_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::rename_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void clear_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::clear_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void mark_flags(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::mark_flags,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void append(Args&&... args)
    {
        call_with_rc<mb::imap_id_uidvalidity_pair>(
            context_, rc_storage_, &mailbox::append, make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void send(Args&&... args)
    {
        call_with_rc<>(
            context_, rc_storage_, &mailbox::send, make_args_tuple(std::forward<Args>(args)...));
    }

    bool authenticated()
    {
        return ext_mb_->authenticated();
    }

    void reset()
    {
        return ext_mb_->reset();
    }

    mb::external::statistics get_stats()
    {
        return ext_mb_->get_stats();
    }

private:
    template <typename... Args>
    auto make_args_tuple(Args&&... args)
    {
        return std::tuple<mailbox*, std::decay_t<Args>...>(
            ext_mb_.get(), std::forward<Args>(args)...);
    }

    context_ptr context_;
    std::shared_ptr<mailbox> ext_mb_;
    rate_controllers_storage rc_storage_;
};

using ext_mailbox_ptr = std::shared_ptr<ext_mailbox>;

}
