#pragma once

#include "common.h"

#include <mailbox/local/local_mailbox_impl.h>

namespace xeno::rc {

namespace mb = mailbox;

/**
 * rc::loc_mailbox proxy all requests to loc_mailbox through rate_controller
 */
template <typename LocalMailboxImpl>
class loc_mailbox : public std::enable_shared_from_this<loc_mailbox<LocalMailboxImpl>>
{
public:
    using mailbox = LocalMailboxImpl;

    loc_mailbox(context_ptr context, std::shared_ptr<mailbox> loc_mb, const std::string& shard_id)
        : context_(context), loc_mb_(loc_mb), rc_storage_("local_mailbox.", "." + shard_id)
    {
    }

    void cancel()
    {
        rc_storage_.cancel(context_);
    }

    template <typename... Args>
    void get_folder_vector(Args&&... args)
    {
        call_with_rc<mb::folder_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_folder_vector,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_messages_info_by_id(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_messages_info_by_id,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_messages_info_chunk_by_id(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_messages_info_chunk_by_id,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_messages_info_top(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_messages_info_top,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_messages_info_without_flags_by_mid(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_messages_info_without_flags_by_mid,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_not_downloaded_messages(Args&&... args)
    {
        call_with_rc<mb::message_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_not_downloaded_messages,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_mids_by_tids(Args&&... args)
    {
        call_with_rc<mb::mid_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_mids_by_tids,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_flags_by_lids(Args&&... args)
    {
        call_with_rc<mb::flags_ptr>(
            context_,
            rc_storage_,
            &mailbox::get_flags_by_lids,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_messages_by_id(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_messages_by_id,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_messages_by_mid(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_messages_by_mid,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void move_messages(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::move_messages,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void init_system_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::init_system_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void create_folder(Args&&... args)
    {
        call_with_rc<mb::folder_vector_ptr>(
            context_,
            rc_storage_,
            &mailbox::create_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void update_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::update_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_mailish_folder_entry(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_mailish_folder_entry,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void clear_folder(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::clear_folder,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void set_folder_symbol(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::set_folder_symbol,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void set_folders_order(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::set_folders_order,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void update_flags(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::update_flags,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void change_flags(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::change_flags,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void update_downloaded_range(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::update_downloaded_range,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void store_message(Args&&... args)
    {
        loc_mb_->store_message(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void check_spam(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::check_spam,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void increment_mailish_entry_errors_count(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::increment_mailish_entry_errors_count,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_mailish_entry(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_mailish_entry,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void clear(Args&&... args)
    {
        call_with_rc<>(
            context_, rc_storage_, &mailbox::clear, make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void erase_security_locks(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::erase_security_locks,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_account(Args&&... args)
    {
        loc_mb_->get_account(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void save_account(Args&&... args)
    {
        loc_mb_->save_account(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void invalidate_auth_data(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::invalidate_auth_data,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void compose_draft(Args&&... args)
    {
        call_with_rc<json::value>(
            context_,
            rc_storage_,
            &mailbox::compose_draft,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void compose_message(Args&&... args)
    {
        call_with_rc<json::value>(
            context_,
            rc_storage_,
            &mailbox::compose_message,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_attachments_sids(Args&&... args)
    {
        call_with_rc<std::vector<std::string>>(
            context_,
            rc_storage_,
            &mailbox::get_attachments_sids,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void update_last_sync_ts(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::update_last_sync_ts,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_karma(Args&&... args)
    {
        call_with_rc<karma_t>(
            context_,
            rc_storage_,
            &mailbox::get_karma,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_or_create_label(Args&&... args)
    {
        call_with_rc<mb::lid>(
            context_,
            rc_storage_,
            &mailbox::get_or_create_label,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_or_create_label_by_symbol(Args&&... args)
    {
        call_with_rc<mb::lid>(
            context_,
            rc_storage_,
            &mailbox::get_or_create_label_by_symbol,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void update_label(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::update_label,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void delete_label(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::delete_label,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void get_send_operation_result(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::get_send_operation_result,
            make_args_tuple(std::forward<Args>(args)...));
    }

    template <typename... Args>
    void save_send_operation_result(Args&&... args)
    {
        call_with_rc<>(
            context_,
            rc_storage_,
            &mailbox::save_send_operation_result,
            make_args_tuple(std::forward<Args>(args)...));
    }

private:
    template <typename... Args>
    auto make_args_tuple(Args&&... args)
    {
        return std::tuple<mailbox*, std::decay_t<Args>...>(
            loc_mb_.get(), std::forward<Args>(args)...);
    }

    context_ptr context_;
    std::shared_ptr<mailbox> loc_mb_;
    rate_controllers_storage rc_storage_;
};

template <typename LocalMailboxImpl>
using loc_mailbox_ptr = std::shared_ptr<loc_mailbox<LocalMailboxImpl>>;

}
