#pragma once

#include "xeno_settings.h"
#include "operations/environment.h"
#include "operations/sync/main_op.h"

#include <auth/get_access_token_op.h>
#include <common/account.h>
#include <common/find_deps.h>
#include <mailbox/data_types/cache_mailbox.h>
#include <mailbox/data_types/sync_status.h>
#include <mailbox/external/external_mailbox_impl.h>
#include <mailbox/local/local_mailbox_impl.h>
#include <mdb/errors.h>
#include <mdb/service.h>
#include <xiva/xiva.h>

#include <yplatform/task_context.h>
#include <yplatform/ptree.h>
#include <yplatform/time_traits.h>
#include <yplatform/log.h>
#include <yplatform/util/sstream.h>

#include <boost/asio/io_service.hpp>

#include <memory>
#include <queue>

namespace xeno {

enum class sync_state
{
    initial,
    running_iteration,
    running_user_op,
    iteration_finished,
    no_auth_data,
    finished
};

inline std::string to_string(sync_state state)
{
    switch (state)
    {
    case sync_state::initial:
        return "initial";
    case sync_state::running_iteration:
        return "running_iteration";
    case sync_state::running_user_op:
        return "running_user_op";
    case sync_state::iteration_finished:
        return "iteration_finished";
    case sync_state::no_auth_data:
        return "no_auth_data";
    case sync_state::finished:
        return "finished";
    default:
        return "undefined";
    }
}

struct controller_dump
{
    uid_t uid;
    std::string email;
    std::string imap_host;
    std::string provider;
    sync_state state;
    std::size_t user_operations_queue_size;
    std::time_t creation_time;
    time_traits::duration sync_newest_inbox_lag;
    time_traits::duration sync_newest_total_lag;

    yplatform::ptree to_ptree() const
    {
        yplatform::ptree ret;
        ret.put("uid", uid);
        ret.put("email", email);
        ret.put("imap_server", imap_host);
        ret.put("providers", provider);
        ret.put("state", to_string(state));
        ret.put("user_operations_queue_size", user_operations_queue_size);
        ret.put("creation_time", creation_time);
        ret.put("sync_newest_inbox_lag", sync_newest_inbox_lag);
        ret.put("sync_newest_total_lag", sync_newest_total_lag);
        return ret;
    }
};

template <typename ExternalMb, typename LocalMb, typename MainOp>
class synchronization_controller_impl
    : public yplatform::log::contains_logger
    , public std::enable_shared_from_this<
          synchronization_controller_impl<ExternalMb, LocalMb, MainOp>>
{
public:
    synchronization_controller_impl(
        boost::asio::io_service* io,
        xeno_settings_ptr settings,
        LocalMb local_mailbox,
        ExternalMb ext_mailbox,
        mailbox::cache_mailbox_ptr mailbox_cache,
        yplatform::task_context_ptr context)
        : yplatform::log::contains_logger(context->logger())
        , io_(io)
        , context_(context)
        , settings_(settings)
        , local_mailbox_(local_mailbox)
        , external_mailbox_(ext_mailbox)
        , mailbox_cache_(mailbox_cache)
        , timer_(*io)
        , typed_log_(find_typed_log())
    {
        if (!mailbox_cache)
        {
            throw std::runtime_error(
                "cannot construct synchronization_controller with empty cache");
        }
    }

    // Interruption handler.
    template <typename Environment, typename... Args>
    void handle_operation_interrupt(error error, Environment&& env, Args&&... args)
    {
        try
        {
            ext_mb_stats_[env.sync_phase] = external_mailbox_->get_stats();

            if (error == mdb::database_errors::user_removed)
            {
                YLOG_L(info) << "finish controller - user removed";
                state_ = sync_state::finished;
                env.mark_handled();
                return handle_operation_finish(env.stat, error);
            }

            if (iteration_deadline_ < yplatform::time_traits::clock::now())
            {
                YLOG_L(info) << "finishing iteration - iteration deadline";
                state_ = sync_state::iteration_finished;
                env.mark_handled();
                return handle_operation_finish(env.stat, error);
            }

            auto& handler = env.get_operation_handler();
            return handler(error, std::forward<Args>(args)...);
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "finishing iteration - interruption handler exception: " << e.what();
            env.mark_handled();
            handle_operation_finish(env.stat, error);
        }
    };

    void handle_operation_finish(iteration_stat_ptr stat, error err)
    {
        stat->finish_phase();

        if (err == code::no_auth_data)
        {
            YLOG_L(error) << "stopping controller: no auth data left";
            state_ = sync_state::no_auth_data;
        }

        if (state_ == sync_state::running_user_op)
        {
            YLOG_L(info) << "user operation finished: " << err.message();
            state_ = sync_state::iteration_finished;
        }

        if (state_ == sync_state::running_iteration)
        {
            YLOG_L(info) << "iteration finished: " << err.message();
            if (need_log_iteration_result(stat, err))
            {
                log_iteration_result(stat, err);
            }
            state_ = sync_state::iteration_finished;
        }

        if (!user_operations_.empty() && external_mailbox_->authenticated())
        {
            return run_user_op();
        }

        auto& background_sync = settings_->sync_settings->background_sync;
        auto iteration_start_time = err == code::auth_tmp_error ?
            time_traits::clock::now() + background_sync.auth_retry_interval :
            iteration_deadline_;
        if (iteration_start_time >= time_traits::clock::now())
        {
            wait_until(iteration_start_time, [this, self = weak_from_this()]() {
                if (self.lock())
                {
                    run_next_operation();
                }
            });
        }
        else
        {
            run_next_operation();
        }
    }

    void start()
    {
        run_next_operation();
    }

    template <typename Operation, typename Callback>
    void add_user_operation(Operation&& op, const Callback& cb)
    {
        // TODO : invoke callback with error if controller is in no_auth_data state
        auto stat = std::make_shared<iteration_stat>();
        auto self = this->shared_from_this();
        auto env = prepare_uninterruptible_env(
            stat, io_->wrap([stat, cb, this, self](error e, auto&&... args) {
                yplatform::safe_call(cb, e, args...);
                handle_operation_finish(stat, e);
            }));
        env.sync_phase = sync_phase::user_op;
        env.deadline = time_traits::clock::now() + settings_->user_operation_timeout;
        auto wrapped_op = std::bind(op, env);

        io_->post([this, self, wrapped_op]() {
            user_operations_.push(wrapped_op);

            if (state_ == sync_state::running_iteration)
            {
                reset_iteration();
            }
            else if (state_ == sync_state::iteration_finished && external_mailbox_->authenticated())
            {
                run_user_op();
            }
        });
    }

    std::weak_ptr<synchronization_controller_impl> weak_from_this()
    {
        return this->shared_from_this();
    }

    void update_account()
    {
        io_->post([this, self = this->shared_from_this()] {
            local_mailbox_->get_account(io_->wrap([this, self](error err, account_t account) {
                if (err)
                {
                    YLOG_L(error) << "synchronization_controller update account error: "
                                  << err.message();
                    return;
                }
                YLOG_L(info) << "updating account";
                mailbox_cache_->update_account(account);
                if (state_ == sync_state::no_auth_data)
                {
                    run_iteration();
                }
            }));
        });
    }

    void update_karma(const karma_t karma)
    {
        io_->post([this, self = this->shared_from_this(), karma] {
            YLOG_L(info) << "update karma value=" << karma.value << " status=" << karma.status;
            auto& account = mailbox_cache_->account();
            account.karma = karma;
        });
    }

    template <typename Callback>
    void update_account_uuid(
        const std::string& xtoken_id,
        const std::string& uuid,
        const Callback& cb)
    {
        io_->post([this, self = this->shared_from_this(), xtoken_id, uuid, cb] {
            auto& account = mailbox_cache_->account();
            auto it = std::find(account.auth_data.begin(), account.auth_data.end(), xtoken_id);
            if (it == account.auth_data.end())
            {
                YLOG_L(error) << "synchronization_controller update uuid error: not found xtoken";
                return cb(code::cannot_unsubscribe_user);
            }
            if (it->uuid == uuid)
            {
                return cb(code::ok);
            }
            if (it->uuid.size())
            {
                io_->post([this, self, uid = account.uid, uuid = it->uuid] {
                    auto xiva_settings = settings_->sync_settings->xiva_settings;
                    unsubscribe(uid, uuid, context_, xiva_settings, [this, self](error ec) {
                        if (ec)
                        {
                            YLOG_L(error) << "synchronization_controller cannot unsubscribe user: "
                                          << ec.message();
                        }
                    });
                });
            }

            auto auth_data = *it;
            auth_data.uuid = uuid;
            auto account_copy = account;
            account_copy.auth_data.clear();
            account_copy.auth_data.push_back(std::move(auth_data));

            local_mailbox_->save_account(
                std::move(account_copy), [this, self, xtoken_id, uuid, cb](error err) {
                    if (err)
                    {
                        return cb(err);
                    }
                    auto& account = mailbox_cache_->account();
                    auto it =
                        std::find(account.auth_data.begin(), account.auth_data.end(), xtoken_id);
                    if (it != account.auth_data.end())
                    {
                        it->uuid = uuid;
                    }
                    else
                    {
                        YLOG_L(warning) << "synchronization_controller update uuid warning: not "
                                           "found xtoken in cache after update in local mailbox";
                    }
                    cb(code::ok);
                });
        });
    }

    template <typename Callback>
    void cache_dump(const Callback& cb) const
    {
        auto self = this->shared_from_this();
        io_->post([this, self, cb = cb]() { cb(code::ok, mailbox_cache_->dump()); });
    }

    template <typename Callback>
    void get_folders(const Callback& cb)
    {
        io_->post([this, self = this->shared_from_this(), cb] {
            auto folders = mailbox_cache_->folders_copy();
            cb({}, folders);
        });
    }

    template <typename Callback>
    void is_folders_locked_for_api_read(const mailbox::fid_vector& fids, const Callback& cb) const
    {
        auto self = this->shared_from_this();
        io_->post([this, self, fids, cb]() {
            error ec = code::ok;
            bool locked = false;

            for (auto& fid : fids)
            {
                auto folder = mailbox_cache_->get_folder_by_fid(fid);
                if (!folder)
                {
                    if (mailbox_cache_->is_not_inited_system_folder(fid))
                    {
                        continue;
                    }
                    ec = code::folder_not_found;
                    break;
                }

                if (folder->api_read_lock)
                {
                    locked = true;
                    break;
                }
            }
            cb(ec, locked);
        });
    }

    template <typename Callback>
    void get_sync_status(const Callback& cb)
    {
        io_->post([this, self = this->shared_from_this(), cb] { cb({}, get_sync_status()); });
    }

    mailbox::sync_status get_sync_status()
    {
        mailbox::sync_status res;
        auto newest_state = mailbox_cache_->sync_newest_state();
        for (auto& [path, folder] : mailbox_cache_->folders())
        {
            mailbox::sync_status::folder_status status;
            status.external_messages_count = folder.count;
            status.api_read_lock = folder.api_read_lock;
            status.turbo_sync_running =
                newest_state->turbo_sync.enabled(folder, settings_->sync_settings->turbo_sync);
            res.folders[folder.fid] = status;
        }

        auto inbox = mailbox_cache_->get_folder_by_type(mailbox::folder::type_t::inbox);
        res.inbox_sync_lag = sync_newest_inbox_lag();
        res.total_sync_lag = sync_newest_total_lag();
        return res;
    }

    void is_finished(const std::function<void(bool)>& cb) const
    {
        io_->post(
            [this, self = this->shared_from_this(), cb] { cb(state_ == sync_state::finished); });
    }

    // This is not thread safe function
    controller_dump dump()
    {
        controller_dump ret;
        auto& account = mailbox_cache_->account();
        ret.uid = account.uid;
        ret.email = account.email;
        ret.imap_host = account.imap_ep.host;
        ret.provider = external_mailbox_->get_provider_unsafe();
        ret.state = state_;
        ret.user_operations_queue_size = user_operations_.size();
        ret.creation_time = creation_time;
        ret.sync_newest_inbox_lag = sync_newest_inbox_lag();
        ret.sync_newest_total_lag = sync_newest_total_lag();
        return ret;
    }

    boost::asio::io_service* io()
    {
        return io_;
    }

private:
    template <typename Handler>
    auto prepare_interrupptible_env(iteration_stat_ptr stat, const Handler& handler)
    {
        auto env = make_env<synchronization_controller_impl, const Handler&, ExternalMb, LocalMb>(
            io_, context_, logger(), stat, this->shared_from_this(), handler);

        env.ext_mailbox = external_mailbox_;
        env.loc_mailbox = local_mailbox_;
        env.cache_mailbox = mailbox_cache_;
        env.sync_settings = settings_->sync_settings;

        return env;
    }

    template <typename Handler>
    auto prepare_uninterruptible_env(iteration_stat_ptr stat, const Handler& handler)
    {
        auto env =
            make_env<const Handler&, ExternalMb, LocalMb>(io_, context_, logger(), stat, handler);

        env.ext_mailbox = external_mailbox_;
        env.loc_mailbox = local_mailbox_;
        env.cache_mailbox = mailbox_cache_;
        env.sync_settings = settings_->sync_settings;

        return env;
    }

    template <typename Handler>
    void wait_until(time_traits::time_point point, Handler&& h)
    {
        timer_.expires_at(point);
        timer_.async_wait([handler = std::forward<Handler>(h)](const error& ec) {
            if (ec == boost::asio::error::operation_aborted)
            {
                return;
            }
            handler();
        });
    }

    bool need_log_iteration_result(iteration_stat_ptr stat, error err)
    {
        if (err) return true;

        if (stat->store_message_attempts > 0) return true;
        if (stat->downloaded_range_updated > 0) return true;

        if (last_iteration_result_log_time + settings_->max_delay_between_iteration_result_logs <=
            time_traits::clock::now())
            return true;

        return false;
    }

    void log_iteration_result(iteration_stat_ptr stat, error err)
    {
        last_iteration_result_log_time = time_traits::clock::now();
        typed_log_->iteration_result(context_, mailbox_cache_->account().uid, stat, err);
    }

    void run_next_operation()
    {
        if (state_ != sync_state::initial && state_ != sync_state::iteration_finished) return;

        if (!user_operations_.empty() && external_mailbox_->authenticated())
        {
            return run_user_op();
        }
        run_iteration();
    }

    void run_iteration()
    {
        if (need_update_token())
        {
            update_access_token(
                [this, self = this->shared_from_this()](error /*err*/) { run_main_op(); });
        }
        else
        {
            run_main_op();
        }
    }

    void run_main_op()
    {
        YLOG_L(info) << "iteration started";
        state_ = sync_state::running_iteration;
        iteration_deadline_ = time_traits::clock::now() + settings_->iteration_timeout;
        auto stat = std::make_shared<iteration_stat>();
        auto weak_this = weak_from_this();
        auto env =
            prepare_interrupptible_env(stat, [this, weak_this, ctx = context_, stat](error error) {
                if (auto locked = weak_this.lock())
                {
                    handle_operation_finish(stat, error);
                }
                else
                {
                    YLOG_CTX_GLOBAL(ctx, info) << "synchronization controller destroyed";
                }
            });
        env.deadline = iteration_deadline_;

        if (ext_mb_stats_.size())
        {
            log_stats();
            ext_mb_stats_.clear();
        }
        ext_mb_stats_[sync_phase::initial] = external_mailbox_->get_stats();

        spawn<MainOp>(std::move(env));
    }

    void run_user_op()
    {
        if (user_operations_.empty()) return;

        state_ = sync_state::running_user_op;

        auto op = std::move(user_operations_.front());
        user_operations_.pop();

        YLOG_L(info) << "user operation started";
        return op();
    }

    void reset_iteration()
    {
        YLOG_L(info) << "reset iteration";
        iteration_deadline_ = yplatform::time_traits::clock::now();
        external_mailbox_->cancel();
        local_mailbox_->cancel();
    }

    void log_stats()
    {
        if (ext_mb_stats_.size() < 2)
        {
            return;
        }
        std::stringstream stream;
        stream << "sync statistics:";
        auto it = ext_mb_stats_.begin();
        bool first = true;
        for (auto prev_it = it++; it != ext_mb_stats_.end(); ++it, ++prev_it)
        {
            auto& cur_stats = it->second;
            auto& prev_stats = prev_it->second;
            stream << (first ? "" : ",") << " phase=" << xeno::to_string(it->first)
                   << " ext_mb_commands=" << cur_stats.command_count - prev_stats.command_count
                   << " ext_mb_rx=" << cur_stats.received_bytes - prev_stats.received_bytes
                   << " ext_mb_tx=" << cur_stats.sent_bytes - prev_stats.sent_bytes;
            first = false;
        }
        YLOG_L(info) << stream.str();
    }

    bool need_update_token()
    {
        if (!settings_->background_access_tokens_loading ||
            time_traits::clock::now() < access_token_expire_ts_ || loading_access_token_)
        {
            return false;
        }
        auto& auth_data_list = mailbox_cache_->account().auth_data;
        if (auth_data_list.empty() || auth_data_list.front().type != auth_type::oauth)
        {
            return false;
        }
        return true;
    }

    void update_access_token(const std::function<void(error)>& cb)
    {
        auto& auth_data = mailbox_cache_->account().auth_data;
        if (auth_data.empty())
        {
            return cb(code::no_auth_data);
        }

        YLOG_L(info) << "update access token";
        auto& auth = auth_data.front();
        auto get_token_cb =
            io_->wrap([this, self = this->shared_from_this(), cb](
                          error err, const std::string&, const time_point& expires_at) {
                if (!err)
                {
                    access_token_expire_ts_ = expires_at;
                }
                loading_access_token_ = false;
                cb(code::ok);
            });
        loading_access_token_ = true;
        bool skip_cache = access_token_expire_ts_ != time_traits::time_point::min();
        yplatform::spawn(std::make_shared<auth::get_access_token_op>(
            context_,
            auth.oauth_app,
            auth.imap_credentials,
            settings_->ext_mailbox_settings->social_settings,
            skip_cache,
            get_token_cb));
    }

    time_traits::duration sync_newest_inbox_lag()
    {
        auto newest_state = mailbox_cache_->sync_newest_state();
        auto inbox = mailbox_cache_->get_folder_by_type(mailbox::folder::type_t::inbox);
        return inbox ? newest_state->get_folder_sync_lag(inbox->path) :
                       time_traits::clock::now() - newest_state->creation_ts_;
    }

    time_traits::duration sync_newest_total_lag()
    {
        return mailbox_cache_->sync_newest_state()->get_total_sync_lag();
    }

private:
    boost::asio::io_service* io_;
    yplatform::task_context_ptr context_;
    xeno_settings_ptr settings_;

    LocalMb local_mailbox_;
    ExternalMb external_mailbox_;
    mailbox::cache_mailbox_ptr mailbox_cache_;

    yplatform::time_traits::time_point iteration_deadline_;

    yplatform::time_traits::timer timer_;
    sync_state state_{ sync_state::initial };

    std::queue<std::function<void()>> user_operations_;
    std::time_t creation_time = std::time(nullptr);
    time_traits::time_point access_token_expire_ts_ = time_traits::time_point::min();
    bool loading_access_token_ = false;
    std::map<sync_phase::sync_phase_t, mailbox::external::statistics> ext_mb_stats_;

    typed_log_ptr typed_log_;
    time_traits::time_point last_iteration_result_log_time = time_traits::time_point::min();
};

namespace detail {

using loc_mailbox_type = mailbox::local::local_mailbox_impl<mdb::ServicePtr>;
using loc_mailbox_rc_type = rc::loc_mailbox_ptr<loc_mailbox_type>;

}

using synchronization_controller =
    synchronization_controller_impl<rc::ext_mailbox_ptr, detail::loc_mailbox_rc_type, main_op>;
using operation_controller_ptr = std::shared_ptr<synchronization_controller>;

}

#include <boost/asio/unyield.hpp>
