#pragma once

#include <common/types.h>
#include <common/account.h>
#include <common/json.h>
#include <common/karma.h>
#include <mailbox/data_types/folder.h>
#include <mailbox/common.h>
#include <mailbox/data_types/store_request.h>
#include <mailbox/data_types/sync_status.h>
#include <mailbox/external/external_mailbox_impl.h>

#include <yplatform/module.h>
#include <yplatform/ptree.h>

namespace xeno {

using uid_t = uid_t;
using mid_t = mailbox::mid_t;
using fid_t = mailbox::fid_t;
using fid_vector = mailbox::fid_vector;
using fid_t_opt = mailbox::fid_t_opt;
using tab_opt = mailbox::tab_opt;
using mid_vector = mailbox::mid_vector;
using tid_vector = mailbox::tid_vector;
using store_request_ptr = store_request_ptr;
using send_request_ptr = send_request_ptr;
using lid = mailbox::lid;
using lid_vector = mailbox::lid_vector;
using ext_mailbox_ptr = std::shared_ptr<mailbox::external::ext_mailbox_impl>;
using sync_status = mailbox::sync_status;

using ptree_cb = std::function<void(error, yplatform::ptree)>;
using json_cb = std::function<void(error, Json::Value)>;
using folders_locked_cb = std::function<void(error, bool)>;
using sync_status_cb = std::function<void(error, const mailbox::sync_status&)>;

class xeno : public yplatform::module
{
public:
    virtual void create_folder(
        uid_t uid,
        const std::string& name,
        const fid_t& parent_fid,
        const std::string& symbol,
        const mailbox::fid_cb& cb) = 0;
    virtual void clear_folder(uid_t uid, const fid_t& fid, const without_data_cb& cb) = 0;
    virtual void delete_folder(uid_t uid, const fid_t& fid, const without_data_cb& cb) = 0;
    virtual void update_folder(
        uid_t uid,
        const fid_t& fid,
        const std::string& name,
        const fid_t_opt& parent_fid,
        const without_data_cb& cb) = 0;
    virtual void get_folders(uid_t uid, const mailbox::folder_vector_cb& cb) = 0;
    virtual void set_folder_symbol(
        uid_t uid,
        const fid_t& fid,
        const std::string& symbol,
        const without_data_cb& cb) = 0;
    virtual void set_folders_order(
        uid_t uid,
        const fid_t& fid,
        const fid_t& prev_fid,
        const without_data_cb& cb) = 0;

    virtual void set_messages_read_flag(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        bool read,
        const without_data_cb& cb) = 0;
    virtual void set_messages_label(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        const lid_vector& lids,
        bool set,
        const without_data_cb& cb) = 0;
    virtual void delete_messages(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        bool purge,
        const without_data_cb& cb) = 0;
    virtual void move_messages(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        const fid_t& dst_fid,
        const tab_opt& dst_tab,
        const without_data_cb& cb) = 0;
    virtual void set_spam_status(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        bool spam,
        const without_data_cb& cb) = 0;

    virtual void get_or_create_label(
        uid_t uid,
        const std::string& name,
        const std::string& color,
        const std::string& type,
        bool force_create,
        const mailbox::lid_cb& cb) = 0;
    virtual void get_or_create_label_by_symbol(
        uid_t uid,
        const std::string& symbol,
        bool force_create,
        const mailbox::lid_cb& cb) = 0;
    virtual void update_label(
        uid_t uid,
        const lid& lid,
        const std::string& name,
        const std::string& color,
        const without_data_cb& cb) = 0;
    virtual void delete_label(uid_t uid, const lid& lid, const without_data_cb& cb) = 0;

    virtual void compose_and_save_draft(
        uid_t uid,
        const std::string& user_ticket,
        store_request_ptr request,
        const json_cb& cb) = 0;
    virtual void save_draft(
        uid_t uid,
        const fid_t& fid,
        std::string&& body,
        const mailbox::mid_cb& cb) = 0;
    virtual void compose_and_send(
        uid_t uid,
        const std::string& user_ticket,
        send_request_ptr request,
        const bool_cb& cb) = 0;
    virtual void send(
        uid_t uid,
        const std::string& email,
        const std::vector<std::string>& rcpts,
        const std::string& ip,
        const std::string& request_id,
        std::string&& body,
        bool notify,
        const mailbox::mid_cb& cb) = 0;

    virtual void cache_dump(uid_t uid, const ptree_cb& cb) = 0;
    virtual void list_controllers(context_ptr ctx, const ptree_cb& cb) = 0;
    virtual void get_providers_state(context_ptr ctx, const ptree_cb& cb) = 0;

    virtual void get_provider(uid_t uid, const mailbox::provider_cb& cb) = 0;
    virtual void load_user(uid_t uid, const shard_id& shard_id) = 0;
    virtual void unload_user(uid_t uid) = 0;
    virtual void invalidate_user_auth(uid_t uid, const without_data_cb& cb) = 0;
    virtual void update_user_uuid(
        uid_t uid,
        const std::string& xtoken_id,
        const std::string& uuid,
        const without_data_cb& cb) = 0;
    virtual void update_user_karma(
        const shard_id& shard_id,
        uid_t uid,
        const karma_t& karma,
        const without_data_cb& cb) = 0;
    virtual void is_folders_locked_for_api_read(
        uid_t uid,
        const fid_vector& fids,
        const folders_locked_cb& cb) = 0;
    virtual void get_sync_status(uid_t uid, const sync_status_cb&) = 0;
    virtual std::map<shard_id, std::size_t> get_users_distribution() = 0;

    virtual ext_mailbox_ptr make_external_mailbox(
        context_ptr ctx,
        const yplatform::log::source& logger) = 0;
};

} // namespace xeno
