#include "xeno_impl.h"
#include "dump_controllers_op.h"
#include "load_cache_op.h"
#include "rc/external_mailbox.h"
#include "rc/local_mailbox.h"
#include "operations/user/create_folder_op.h"
#include "operations/user/clear_folder_op.h"
#include "operations/user/delete_folder_with_subfolders_op.h"
#include "operations/user/update_folder_op.h"
#include "operations/user/set_read_flag_op.h"
#include "operations/user/delete_messages_op.h"
#include "operations/user/move_messages_op.h"
#include "operations/user/compose_and_save_draft_op.h"
#include "operations/user/save_draft_op.h"
#include "operations/user/compose_and_send_op.h"
#include "operations/user/send_op.h"
#include "operations/user/set_spam_status_op.h"
#include "operations/user/set_label_op.h"
#include "operations/user/get_provider_op.h"
#include "operations/user/get_or_create_label_op.h"
#include "operations/user/update_label_op.h"
#include "operations/user/delete_label_op.h"
#include "operations/user/get_or_create_label_by_symbol_op.h"
#include "operations/user/set_folder_symbol_op.h"
#include "operations/user/set_folders_order_op.h"
#include "operations/user/invalidate_user_auth_op.h"

#include <ymod_mdb_sharder/users_distributor.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/util/execution_holder.h>
#include <yplatform/module_registration.h>
#include <iostream>
#include <mdb/uid_resolver.h>
#include <mdb/users_polling.h>

#include <auth/social/settings.h>
#include <mailbox/local/local_mailbox_impl.h>

#include <xiva/xiva_settings.h>

using namespace xeno::mailbox;

namespace xeno {

namespace ph = std::placeholders;

xeno_impl::xeno_impl(yplatform::reactor& reactor) : reactor_(reactor)
{
    for (size_t i = 0; i < reactor_.size(); ++i)
    {
        if (reactor_[i]->size() != 1)
            throw std::runtime_error("xeno_impl is optimized for single-thread reactors - set "
                                     "pool_count=N and io_threads=1");
    }
}

void xeno_impl::init(const yplatform::ptree& conf)
{
    settings_->iteration_timeout =
        conf.get<time_traits::duration>("iteration_timeout", time_traits::seconds(10));
    settings_->user_operation_timeout =
        conf.get<time_traits::duration>("user_operation_timeout", time_traits::seconds(30));
    settings_->cleanup_controllers_interval =
        conf.get<time_traits::duration>("cleanup_controllers_interval", time_traits::seconds(10));
    settings_->load_account_retry_interval =
        conf.get<time_traits::duration>("load_account_retry_interval", time_traits::seconds(2));
    settings_->max_delay_between_iteration_result_logs = conf.get<time_traits::duration>(
        "max_delay_between_iteration_result_logs", time_traits::hours(1));
    settings_->background_access_tokens_loading =
        conf.get<bool>("background_access_tokens_loading", false);
    settings_->attachments_url = conf.get<std::string>("webattach.url");
    for (auto& [name, uid] : boost::make_iterator_range(conf.equal_range("forbidden_uids")))
    {
        settings_->forbidden_uids.insert(std::stoull(uid.data()));
    }

    mailbox::local::settings local_mb_settings;
    local_mb_settings.delete_messages_chunk = conf.get("delete_messages_chunk", 50);

    settings_->loc_mailbox_settings =
        std::make_shared<const mailbox::local::settings>(std::move(local_mb_settings));
    settings_->ext_mailbox_settings->update(conf.get_child("mailbox.external"));

    synchronization_settings sync_settings;
    sync_settings.delete_messages_chunk_size = local_mb_settings.delete_messages_chunk;
    sync_settings.newest_count = conf.get<uint32_t>("sync_messages.newest_count", 20);
    sync_settings.oldest_count = conf.get<uint32_t>("sync_messages.oldest_count", 10);
    sync_settings.oldest_cache_size = conf.get<uint32_t>("sync_messages.oldest_cache_size", 100);
    sync_settings.oldest_flags_and_deletions_chunk_size =
        conf.get<uint32_t>("sync_messages.oldest_flags_and_deletions_chunk_size", 100);
    sync_settings.user_op_chunk_size = conf.get<uint32_t>("user_op_chunk_size", 100);
    sync_settings.newest_downloading_retries =
        conf.get<uint32_t>("sync_messages.newest_downloading_retries", 3);
    yplatform::read_ptree(sync_settings.low_priority_providers, conf, "low_priority_providers");
    auto sync_messages_settings = conf.get_child("sync_messages");
    for (auto& [name, entry] :
         boost::make_iterator_range(sync_messages_settings.equal_range("turbo_sync")))
    {
        auto symbol = mailbox::folder::type_from_string(entry.get<string>("folder_type"));
        sync_settings.turbo_sync.folders[symbol].count = entry.get<uint32_t>("count");
        sync_settings.turbo_sync.folders[symbol].timeout =
            entry.get<time_traits::duration>("timeout");
    }
    sync_settings.redownload_messages_cache_size =
        conf.get<uint32_t>("sync_messages.redownload_messages.cache_size", 10);
    auto redownload_messages_settings = conf.get_child("sync_messages.redownload_messages");
    auto delay_groups = redownload_messages_settings.equal_range("groups");
    for (auto& [name, entry] : boost::make_iterator_range(delay_groups))
    {
        auto threshold = entry.get<uint32_t>("error_count");
        sync_settings.redownload_messages_delay_groups[threshold] =
            entry.get<time_traits::duration>("delay");
    }
    sync_settings.redownload_messages_delay_groups[std::numeric_limits<uint32_t>::max()] =
        conf.get<time_traits::duration>(
            "sync_messages.redownload_messages.max_delay", time_traits::hours(24));
    sync_settings.max_message_size = conf.get<size_t>("sync_messages.max_message_size");

    sync_settings.xiva_settings = std::make_shared<xiva_settings>();
    sync_settings.xiva_settings->token = conf.get<std::string>("xiva_api_token");

    auto& background_sync = sync_settings.background_sync;
    background_sync.auth_retry_interval =
        conf.get<time_traits::duration>("background_sync.auth_retry_interval");
    background_sync.auth_by_password_tries =
        conf.get<int>("background_sync.auth_by_password_tries");
    background_sync.auth_by_oauth_tries = conf.get<int>("background_sync.auth_by_oauth_tries");

    settings_->sync_settings =
        std::make_shared<const synchronization_settings>(std::move(sync_settings));

    macs_ = yplatform::find<ymod_macs::module>("macs");

    auto users_distributor =
        yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor");
    users_distributor->set_polling_methods(&mdb::get_all_users, &mdb::get_changed_users);
    users_distributor->subscribe(
        std::bind(&xeno_impl::on_acquire_accounts, shared_from(this), ph::_1, ph::_2),
        std::bind(&xeno_impl::on_release_accounts, shared_from(this), ph::_2));

    accounts_loading_rc_ =
        yplatform::find<ymod_ratecontroller::rate_controller_module>("rate_controller")
            ->get_controller("local_mailbox.accounts_loading");

    cleanup_controllers_timer_ = std::make_unique<time_traits::timer>(*reactor_.io());

    sync_newest_axis_ = histogram_axis{ 40, 10.0f, 86400.0f };
}

void xeno_impl::start()
{
    run_cleanup_controllers_timer();
}

void xeno_impl::create_folder(
    uid_t uid,
    const std::string& name,
    const fid_t& parent_fid,
    const std::string& symbol,
    const mailbox::fid_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, fid_t());
    }

    controller->add_user_operation(user::create_folder_op(name, parent_fid, symbol), std::move(cb));
}

void xeno_impl::clear_folder(uid_t uid, const fid_t& fid, const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::clear_folder_op(fid), std::move(cb));
}

void xeno_impl::delete_folder(uid_t uid, const fid_t& fid, const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::delete_folder_with_subfolders_op(fid), std::move(cb));
}

void xeno_impl::update_folder(
    uid_t uid,
    const fid_t& fid,
    const std::string& name,
    const fid_t_opt& parent_fid,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::update_folder_op(fid, name, parent_fid), std::move(cb));
}

void xeno_impl::get_folders(uid_t uid, const mailbox::folder_vector_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, nullptr);
    }

    controller->get_folders(cb);
}

void xeno_impl::set_folder_symbol(
    uid_t uid,
    const fid_t& fid,
    const std::string& symbol,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::set_folder_symbol_op(fid, symbol), std::move(cb));
}

void xeno_impl::set_folders_order(
    uid_t uid,
    const fid_t& fid,
    const fid_t& prev_fid,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::set_folders_order_op(fid, prev_fid), std::move(cb));
}

void xeno_impl::set_messages_read_flag(
    uid_t uid,
    const mid_vector& mids,
    const tid_vector& tids,
    bool read,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::set_read_flag_op(mids, tids, read), std::move(cb));
}

void xeno_impl::set_messages_label(
    uid_t uid,
    const mid_vector& mids,
    const tid_vector& tids,
    const lid_vector& lids,
    bool set,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::set_label_op(mids, tids, lids, set), std::move(cb));
}

void xeno_impl::delete_messages(
    uid_t uid,
    const mid_vector& mids,
    const tid_vector& tids,
    bool purge,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::delete_messages_op(mids, tids, purge), std::move(cb));
}

void xeno_impl::move_messages(
    uid_t uid,
    const mid_vector& mids,
    const tid_vector& tids,
    const fid_t& dst_fid,
    const tab_opt& dst_tab,
    const without_data_cb& cb)
{
    std::lock_guard<std::mutex> lock(unload_mutex_);
    auto it = working_controllers_map_.find(uid);
    if (it == working_controllers_map_.end())
    {
        return cb(code::user_not_found);
    }

    auto& [working_uid, controller] = *it;
    controller.impl->add_user_operation(
        user::move_messages_op(mids, tids, dst_fid, dst_tab), std::move(cb));
}

void xeno_impl::compose_and_save_draft(
    uid_t uid,
    const std::string& user_ticket,
    store_request_ptr request,
    const json_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, Json::objectValue);
    }

    controller->add_user_operation(
        user::compose_and_save_draft_op(user_ticket, request, settings_->attachments_url),
        std::move(cb));
}

void xeno_impl::save_draft(
    uid_t uid,
    const fid_t& fid,
    std::string&& body,
    const mailbox::mid_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, 0);
    }

    controller->add_user_operation(user::save_draft_op(fid, std::move(body)), std::move(cb));
}

void xeno_impl::set_spam_status(
    uid_t uid,
    const mid_vector& mids,
    const tid_vector& tids,
    bool spam,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::set_spam_status_op(mids, tids, spam), std::move(cb));
}

void xeno_impl::get_or_create_label(
    uid_t uid,
    const std::string& name,
    const std::string& color,
    const std::string& type,
    bool force_create,
    const mailbox::lid_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, lid());
    }

    controller->add_user_operation(
        user::get_or_create_label_op(name, color, type, force_create), std::move(cb));
}

void xeno_impl::get_or_create_label_by_symbol(
    uid_t uid,
    const std::string& symbol,
    bool force_create,
    const mailbox::lid_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, lid());
    }

    controller->add_user_operation(
        user::get_or_create_label_by_symbol_op(symbol, force_create), std::move(cb));
}

void xeno_impl::update_label(
    uid_t uid,
    const lid& lid,
    const std::string& name,
    const std::string& color,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::update_label_op(lid, name, color), std::move(cb));
}

void xeno_impl::delete_label(uid_t uid, const lid& lid, const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }

    controller->add_user_operation(user::delete_label_op(lid), std::move(cb));
}

void xeno_impl::send(
    uid_t uid,
    const std::string& email,
    const std::vector<std::string>& rcpts,
    const std::string& ip,
    const std::string& request_id,
    std::string&& body,
    bool notify,
    const mailbox::mid_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, 0);
    }

    controller->add_user_operation(
        user::send_op(email, rcpts, ip, request_id, std::move(body), notify), std::move(cb));
}

void xeno_impl::compose_and_send(
    uid_t uid,
    const std::string& user_ticket,
    send_request_ptr request,
    const bool_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, false);
    }

    controller->add_user_operation(user::compose_and_send_op(user_ticket, request), std::move(cb));
}

void xeno_impl::start_loading(
    context_ptr ctx,
    error err,
    pending_account_ptr account,
    const completion_handler& on_complete)
{
    auto rc_holder = std::make_shared<yplatform::execution_holder>(on_complete);
    {
        std::lock_guard<std::mutex> lock(unload_mutex_);
        if (account->aborted) return;
    }
    if (err)
    {
        YLOG(ctx->logger(), error) << "account loading error: " << err.message();
        start_loading_with_delay(ctx, account);
        return;
    }
    auto service = std::make_shared<mdb::Service>(ctx, account->uid, account->shard_id);
    auto accounts_repository =
        std::make_shared<mdb::accounts_repository>(ctx, account->uid, account->shard_id, false);

    using local_mailbox_type = mailbox::local::local_mailbox_impl<mdb::ServicePtr>;
    auto local_mailbox = std::make_shared<local_mailbox_type>(
        *reactor_.io(),
        ctx->logger(),
        ctx,
        service,
        accounts_repository,
        account->uid,
        settings_->loc_mailbox_settings);
    auto local_mailbox_rc = std::make_shared<rc::loc_mailbox<local_mailbox_type>>(
        ctx, local_mailbox, account->shard_id);
    load_cache_op loader(
        ctx,
        local_mailbox,
        [this, self = shared_from(this), account, local_mailbox, local_mailbox_rc, ctx, rc_holder](
            error ec, mailbox::cache_mailbox_ptr cache) mutable {
            {
                std::lock_guard<std::mutex> lock(unload_mutex_);
                if (account->aborted)
                {
                    return;
                }

                if (ec)
                {
                    YLOG(ctx->logger(), error) << "cache loading error: " << ec.message();
                    if (ec == errc::user_should_be_loaded)
                    {
                        start_loading_with_delay(ctx, account);
                    }
                    else
                    {
                        pending_accounts_.erase(account->uid);
                    }
                    return;
                }
            }

            operation_controller_ptr controller_impl;
            try
            {
                auto imap_server = cache->account().imap_ep.host;
                ctx->logger().append_log_prefix(imap_server);
                local_mailbox->logger() = ctx->logger();
                YLOG(ctx->logger(), info) << "cache loaded, email=" << cache->account().email;
                auto external_mailbox = make_external_mailbox(ctx, ctx->logger());
                auto external_mailbox_rc =
                    std::make_shared<rc::ext_mailbox>(ctx, external_mailbox, imap_server);
                controller_impl = std::make_shared<synchronization_controller>(
                    reactor_.io(), settings_, local_mailbox_rc, external_mailbox_rc, cache, ctx);
                std::lock_guard<std::mutex> lock(unload_mutex_);
                working_controllers_map_.emplace(
                    account->uid, working_controller{ account->shard_id, controller_impl });
                pending_accounts_.erase(account->uid);
            }
            catch (const std::exception& e)
            {
                YLOG(ctx->logger(), error) << "cache loading exception: " << e.what();
                start_loading_with_delay(ctx, account);
            }

            if (controller_impl)
            {
                controller_impl->start();
            }
        });
    loader();
}

void xeno_impl::start_loading_with_delay(context_ptr ctx, pending_account_ptr account)
{
    auto timer = std::make_shared<time_traits::timer>(*reactor_.io());
    timer->expires_from_now(settings_->load_account_retry_interval);
    timer->async_wait([account, timer, ctx, this, self = shared_from(this)](error ec) {
        if (ec)
        {
            YLOG(ctx->logger(), error) << "cache loading timer error: " << ec.message();
        }
        accounts_loading_rc_->post(
            std::bind(&xeno_impl::start_loading, self, ctx, ph::_1, account, ph::_2));
    });
}

void xeno_impl::on_acquire_accounts(const shard_id& shard_id, const uid_vector& uids)
{
    std::lock_guard<std::mutex> lock(unload_mutex_);
    for (auto uid : uids)
    {
        if (!working_controllers_map_.count(uid) && !pending_accounts_.count(uid))
        {
            auto ctx = boost::make_shared<yplatform::task_context>();
            ctx->logger() = logger();
            ctx->logger().append_log_prefix(std::to_string(uid) + " " + ctx->uniq_id());
            YLOG(ctx->logger(), info) << "pending account, shard_id=" << shard_id;
            auto account =
                std::make_shared<pending_account>(pending_account{ uid, shard_id, false });
            pending_accounts_[uid] = account;
            accounts_loading_rc_->post(std::bind(
                &xeno_impl::start_loading, shared_from(this), ctx, ph::_1, account, ph::_2));
        }
    }
}

void xeno_impl::on_release_accounts(const uid_vector& uids)
{
    std::lock_guard<std::mutex> lock(unload_mutex_);
    for (auto& uid : uids)
    {
        auto released = working_controllers_map_.erase(uid);
        auto it = pending_accounts_.find(uid);
        if (it != pending_accounts_.end())
        {
            auto& [pending_uid, account] = *it;
            account->aborted = true;
            pending_accounts_.erase(it);
            released = true;
        }
        if (released)
        {
            YLOG_L(info) << "release account: " << uid;
        }
    }
}

void xeno_impl::load_user(uid_t uid, const shard_id& shard_id)
{
    auto controller = get_controller_for_uid(uid);
    if (controller)
    {
        controller
            ->update_account(); // we should update account because it may be in no_auth_data state
        YLOG_L(info) << "update account: " << uid;
    }
    else
    {
        on_acquire_accounts(shard_id, { uid });
    }
}

void xeno_impl::cache_dump(uid_t uid, const ptree_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (controller)
    {
        controller->cache_dump(cb);
    }
    else
    {
        return cb(code::user_not_found, yplatform::ptree());
    }
}

void xeno_impl::update_user_uuid(
    uid_t uid,
    const std::string& xtoken_id,
    const std::string& uuid,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (controller)
    {
        controller->update_account_uuid(xtoken_id, uuid, cb);
    }
    else
    {
        cb(code::user_not_found);
    }
}

void xeno_impl::update_user_karma(
    const shard_id& shard_id,
    uid_t uid,
    const karma_t& karma,
    const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (controller)
    {
        if (is_karma_bad(karma))
        {
            YLOG_L(info) << "release account because bad karma: uid=" << uid;
            on_release_accounts({ uid });
        }
        else
        {
            controller->update_karma(karma);
        }
    }
    else if (!is_karma_bad(karma))
    {
        YLOG_L(info) << "acquire account because karma has become a good: uid=" << uid;
        on_acquire_accounts(shard_id, { uid });
    }
    cb(code::ok);
}

ext_mailbox_ptr xeno_impl::make_external_mailbox(
    context_ptr ctx,
    const yplatform::log::source& logger)
{
    using mailbox::external::ext_mailbox_impl;
    return std::make_shared<ext_mailbox_impl>(
        *reactor_.io(), ctx, settings_->ext_mailbox_settings, logger);
}

void xeno_impl::dump_controllers(context_ptr ctx, const controller_dumps_cb& cb) const
{
    std::vector<operation_controller_ptr> controllers;
    {
        std::lock_guard<std::mutex> lock(unload_mutex_);
        controllers.reserve(working_controllers_map_.size());
        for (auto& [uid, controller] : working_controllers_map_)
        {
            controllers.emplace_back(controller.impl);
        }
    }
    yplatform::spawn(std::make_shared<dump_controllers_op>(ctx, controllers, cb));
}

void xeno_impl::list_controllers(context_ptr ctx, const ptree_cb& cb)
{
    dump_controllers(ctx, [cb](error err, const std::vector<controller_dump>& dumps) {
        if (err) return cb(err, {});
        yplatform::ptree dump, controllers;
        for (auto&& dump : dumps)
        {
            controllers.push_back(std::make_pair("", dump.to_ptree()));
        }
        dump.put_child("controllers", controllers);
        cb({}, dump);
    });
}

void xeno_impl::get_providers_state(context_ptr ctx, const ptree_cb& cb)
{
    dump_controllers(ctx, [cb](error err, const std::vector<controller_dump>& dumps) {
        if (err) return cb(err, {});
        std::map<std::string, std::map<sync_state, std::size_t>> stats;
        for (auto&& dump : dumps)
        {
            stats[dump.imap_host][dump.state]++;
        }
        yplatform::ptree ret;
        for (auto&& [imap_server, provider_stats] : stats)
        {
            yplatform::ptree provider_node;
            for (auto&& [state, count] : provider_stats)
            {
                provider_node.put(to_string(state), count);
            }
            ret.push_back(std::make_pair(imap_server, provider_node));
        }
        cb({}, ret);
    });
}

operation_controller_ptr xeno_impl::get_controller_for_uid(uid_t uid)
{
    std::lock_guard<std::mutex> lock(unload_mutex_);
    auto it = working_controllers_map_.find(uid);
    if (it == working_controllers_map_.end())
    {
        return operation_controller_ptr();
    }
    else
    {
        return it->second.impl;
    }
}

void xeno_impl::unload_user(uid_t uid)
{
    on_release_accounts({ uid });
}

void xeno_impl::invalidate_user_auth(uid_t uid, const without_data_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found);
    }
    controller->add_user_operation(user::invalidate_user_auth_op(), cb);
}

void xeno_impl::get_provider(uid_t uid, const mailbox::provider_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (!controller)
    {
        return cb(code::user_not_found, std::string());
    }
    controller->add_user_operation(user::get_provider_op(), std::move(cb));
}

void xeno_impl::is_folders_locked_for_api_read(
    uid_t uid,
    const fid_vector& fids,
    const folders_locked_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (controller)
    {
        controller->is_folders_locked_for_api_read(fids, cb);
    }
    else
    {
        cb(code::user_not_found, true);
    }
}

void xeno_impl::get_sync_status(uid_t uid, const sync_status_cb& cb)
{
    auto controller = get_controller_for_uid(uid);
    if (controller)
    {
        controller->get_sync_status(cb);
    }
    else
    {
        cb(code::user_not_found, {});
    }
}

std::map<shard_id, std::size_t> xeno_impl::get_users_distribution()
{
    std::map<shard_id, std::size_t> ret;
    std::lock_guard<std::mutex> lock(unload_mutex_);
    for (auto& [uid, controller] : working_controllers_map_)
    {
        ++ret[controller.shard_id];
    }
    return ret;
}

void xeno_impl::run_cleanup_controllers_timer()
{
    cleanup_controllers_timer_->expires_from_now(settings_->cleanup_controllers_interval);
    cleanup_controllers_timer_->async_wait(
        std::bind(&xeno_impl::clear_controllers, shared_from(this)));
}

void xeno_impl::clear_controllers()
{
    {
        std::lock_guard<std::mutex> lock(unload_mutex_);
        for (auto& [uid, controller] : working_controllers_map_)
        {
            controller.impl->is_finished(
                [this, self = shared_from_this(), uid = uid](bool finished) {
                    if (finished)
                    {
                        on_release_accounts({ uid });
                    }
                });
        }
    }
    run_cleanup_controllers_timer();
}

}

DEFINE_SERVICE_OBJECT(xeno::xeno_impl);
