#pragma once

#include "synchronization_controller.h"
#include "xeno.h"
#include "xeno_settings.h"

#include <mdb/accounts_repository.h>

#include <yplatform/future/future.hpp>
#include <yplatform/future/multi_future.hpp>
#include <yplatform/util/histogram.h>
#include <deque>
#include <map>
#include <mutex>
#include <numeric>

namespace xeno {

class xeno_impl : public xeno
{
public:
    explicit xeno_impl(yplatform::reactor& reactor);

    void init(const yplatform::ptree& conf);
    void start();
    void create_folder(
        uid_t uid,
        const std::string& name,
        const fid_t& parent_fid,
        const std::string& symbol,
        const mailbox::fid_cb& cb) override;
    void clear_folder(uid_t uid, const fid_t& fid, const without_data_cb& cb) override;
    void delete_folder(uid_t uid, const fid_t& fid, const without_data_cb& cb) override;
    void update_folder(
        uid_t uid,
        const fid_t& fid,
        const std::string& name,
        const fid_t_opt& parent_fid,
        const without_data_cb& cb) override;
    void get_folders(uid_t uid, const mailbox::folder_vector_cb& cb) override;
    void set_folder_symbol(
        uid_t uid,
        const fid_t& fid,
        const std::string& symbol,
        const without_data_cb& cb) override;
    void set_folders_order(
        uid_t uid,
        const fid_t& fid,
        const fid_t& prev_fid,
        const without_data_cb& cb) override;

    void set_messages_read_flag(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        bool read,
        const without_data_cb& cb) override;
    void set_messages_label(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        const lid_vector& lids,
        bool set,
        const without_data_cb& cb) override;
    void delete_messages(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        bool purge,
        const without_data_cb& cb) override;
    void move_messages(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        const fid_t& dst_fid,
        const tab_opt& dst_tab,
        const without_data_cb& cb) override;
    void set_spam_status(
        uid_t uid,
        const mid_vector& mids,
        const tid_vector& tids,
        bool spam,
        const without_data_cb& cb) override;

    void get_or_create_label(
        uid_t uid,
        const std::string& name,
        const std::string& color,
        const std::string& type,
        bool force_create,
        const mailbox::lid_cb& cb) override;
    void get_or_create_label_by_symbol(
        uid_t uid,
        const std::string& symbol,
        bool force_create,
        const mailbox::lid_cb& cb) override;
    void update_label(
        uid_t uid,
        const lid& lid,
        const std::string& name,
        const std::string& color,
        const without_data_cb& cb) override;
    void delete_label(uid_t uid, const lid& lid, const without_data_cb& cb) override;

    void save_draft(uid_t uid, const fid_t& fid, std::string&& body, const mailbox::mid_cb& cb)
        override;
    void compose_and_save_draft(
        uid_t uid,
        const std::string& user_ticket,
        store_request_ptr request,
        const json_cb& cb) override;
    void send(
        uid_t uid,
        const std::string& email,
        const std::vector<std::string>& rcpts,
        const std::string& ip,
        const std::string& request_id,
        std::string&& body,
        bool notify,
        const mailbox::mid_cb& cb) override;
    void compose_and_send(
        uid_t uid,
        const std::string& user_ticket,
        send_request_ptr request,
        const bool_cb& cb) override;

    void cache_dump(uid_t uid, const ptree_cb& cb) override;
    void list_controllers(context_ptr ctx, const ptree_cb& cb) override;
    void get_providers_state(context_ptr ctx, const ptree_cb& cb) override;

    void get_provider(uid_t uid, const mailbox::provider_cb& cb) override;
    void load_user(uid_t uid, const shard_id& shard_id) override;
    void unload_user(uid_t uid) override;
    void invalidate_user_auth(uid_t uid, const without_data_cb& cb) override;
    void update_user_uuid(
        uid_t uid,
        const std::string& xtoken_id,
        const std::string& uuid,
        const without_data_cb& cb) override;
    void update_user_karma(
        const shard_id& shard_id,
        uid_t uid,
        const karma_t& karma,
        const without_data_cb& cb) override;
    void is_folders_locked_for_api_read(
        uid_t uid,
        const fid_vector& fids,
        const folders_locked_cb& cb) override;
    void get_sync_status(uid_t uid, const sync_status_cb&) override;
    std::map<shard_id, std::size_t> get_users_distribution() override;

    ext_mailbox_ptr make_external_mailbox(context_ptr ctx, const yplatform::log::source& logger)
        override;

protected:
    struct pending_account
    {
        uid_t uid;
        ::xeno::shard_id shard_id;
        bool aborted;
    };

    struct working_controller
    {
        ::xeno::shard_id shard_id;
        operation_controller_ptr impl;
    };

    using pending_account_ptr = std::shared_ptr<pending_account>;
    using completion_handler = ymod_ratecontroller::completion_handler;
    using controller_dumps_cb = std::function<void(error, const std::vector<controller_dump>&)>;
    using histogram_axis =
        yplatform::hgram::axis::regular<float, yplatform::hgram::axis::transform::log>;
    using histogram = decltype(yplatform::hgram::make_histogram(std::declval<histogram_axis>()));
    using histogram_map = std::map<std::string, histogram>;

    void start_loading(
        context_ptr ctx,
        error,
        pending_account_ptr account,
        const completion_handler&);
    void start_loading_with_delay(context_ptr ctx, pending_account_ptr account);

    void on_acquire_accounts(const shard_id& shard_id, const uid_vector& uids);
    void on_release_accounts(const uid_vector& uids);

    void run_cleanup_controllers_timer();
    void clear_controllers();
    void dump_controllers(context_ptr ctx, const controller_dumps_cb& cb) const;

    yplatform::future::future_void_t calc_sync_newest_timings_histograms(
        histogram_map& inbox_lag_hgrams,
        histogram_map& total_lag_hgrams) const
    {
        yplatform::future::promise_void_t promise;
        auto ctx = boost::make_shared<yplatform::task_context>();
        dump_controllers(
            ctx,
            [this, self = shared_from(this), &inbox_lag_hgrams, &total_lag_hgrams, promise](
                auto&& ec, auto&& dumps) mutable {
                if (ec)
                {
                    promise.set_exception(std::runtime_error(ec.message()));
                    return;
                }
                for (auto&& dump : dumps)
                {
                    if (dump.state == sync_state::no_auth_data) continue;
                    update_sync_lag_hgram(
                        inbox_lag_hgrams, dump.provider, dump.sync_newest_inbox_lag);
                    update_sync_lag_hgram(
                        total_lag_hgrams, dump.provider, dump.sync_newest_total_lag);
                }
                promise.set();
            });
        return promise;
    }

    void update_sync_lag_hgram(
        histogram_map& hgrams,
        const std::string& provider,
        const time_traits::duration& lag) const
    {
        auto it = hgrams.find(provider);
        if (it == hgrams.end())
        {
            std::tie(it, std::ignore) =
                hgrams.emplace(provider, yplatform::hgram::make_histogram(sync_newest_axis_));
        }
        it->second(time_traits::get_seconds_count(lag));
    }

    yplatform::ptree get_stats() const override
    {
        yplatform::ptree result;
        try
        {
            histogram_map inbox_lag_hgram;
            histogram_map total_lag_hgram;
            calc_sync_newest_timings_histograms(inbox_lag_hgram, total_lag_hgram).get();
            for (auto&& [provider, hgram] : inbox_lag_hgram)
            {
                result.put_child(
                    "sync_newest_inbox_lag_tier_" + provider + "_hgram",
                    yplatform::hgram::to_ptree(hgram));
            }
            for (auto&& [provider, hgram] : total_lag_hgram)
            {
                result.put_child(
                    "sync_newest_total_lag_tier_" + provider + "_hgram",
                    yplatform::hgram::to_ptree(hgram));
            }
        }
        catch (const std::exception& e)
        {
            YLOG_G(error) << "xeno get_stats exception: " << e.what();
        }
        return result;
    }

    operation_controller_ptr get_controller_for_uid(uid_t uid);
    yplatform::reactor& reactor_;
    boost::shared_ptr<ymod_macs::module> macs_;
    xeno_settings_ptr settings_ = std::make_shared<xeno_settings>();

    std::map<uid_t, working_controller> working_controllers_map_;
    std::map<uid_t, pending_account_ptr> pending_accounts_;
    ymod_ratecontroller::rate_controller_ptr accounts_loading_rc_;

    std::unique_ptr<time_traits::timer> cleanup_controllers_timer_;

    std::mutex pending_mutex_;
    mutable std::mutex unload_mutex_;
    histogram_axis sync_newest_axis_;
};

}
