#!/usr/bin/env python3

import argparse
import msgpack
import psycopg2
import yaml


def main():
    argparser = argparse.ArgumentParser()
    argparser.add_argument("platform", help="apns, gcm, hms or wns", type=str)
    argparser.add_argument("app", help="application name", type=str)
    argparser.add_argument("new_owner", help="new owner", type=str)
    argparser.add_argument("--conninfo", help="exact db conninfo", type=str, default="")
    args = argparser.parse_args()

    if args.conninfo:
        conninfo = args.conninfo
    else:
        conninfo = get_conninfo()

    with psycopg2.connect(conninfo) as connection:
        platform_app = args.platform + ":" + args.app
        app_settings = get_app_settings(connection, platform_app)
        app_settings[1] = args.new_owner
        new_owner_id = "xivaservice:" + args.new_owner
        revision = update_app_owner(connection, platform_app, new_owner_id)
        revision = update_app_settings(connection, platform_app, new_owner_id, app_settings)
        print(revision)


def get_conninfo():
    with open("/etc/xivaconf/multienv_conninfo.yml") as f:
        yml = yaml.safe_load(f)
        modules = yml["config"]["modules"]["module"]
        confdb = next(filter(lambda m: m["_name"] == "confdb", modules))
        return confdb["configuration"]["conninfo"]


def get_app_settings(connection, platform_app):
    with connection.cursor() as cursor:
        cursor.execute(SELECT_APP_SETTINGS_SQL, {"platform_app": platform_app})
        result = cursor.fetchall()
        if cursor.rowcount != 1:
            raise Exception("found %s apps instead of 1" % cursor.rowcount)
        packed_settings = result[0][0].tobytes()
        return msgpack.unpackb(packed_settings)


SELECT_APP_SETTINGS_SQL = """
    SELECT
        settings
    FROM
        xconf.configurations
    WHERE
        type = 'mobile'
        AND name = %(platform_app)s
        AND environment = 'any';
"""


def update_app_owner(connection, platform_app, new_owner_id):
    with connection.cursor() as cursor:
        cursor.execute(
            UPDATE_APP_OWNER_ID_SQL,
            {
                "platform_app": platform_app,
                "new_owner_id": new_owner_id,
            },
        )
        result = cursor.fetchall()
        revision, error = result[0]
        if error is not None:
            raise Exception(error)
        return revision


UPDATE_APP_OWNER_ID_SQL = """
    SELECT *
    FROM xconf.update_owner_id('mobile', %(platform_app)s, %(new_owner_id)s, 'any');
"""


def update_app_settings(connection, platform_app, owner_id, new_settings):
    packed_new_settings = msgpack.packb(new_settings)
    with connection.cursor() as cursor:
        cursor.execute(
            UPDATE_APP_SETTINGS_SQL,
            {
                "platform_app": platform_app,
                "owner_id": owner_id,
                "new_settings": packed_new_settings,
            },
        )
        result = cursor.fetchall()
        revision, error = result[0]
        if error is not None:
            raise Exception(error)
        return revision


UPDATE_APP_SETTINGS_SQL = """
    SELECT *
    FROM xconf.put('mobile', %(platform_app)s, %(owner_id)s, %(new_settings)s, NULL, 'any');
"""


if __name__ == "__main__":
    main()
