#include "server.h"
#include <ymod_xconf/pg_client.h>
#include <ymod_xconf/types.h>
#include <ymod_webserver/server.h>
#include <yplatform/find.h>
#include <yplatform/module.h>
#include <stdexcept>

namespace yxiva { namespace conf {

server::server(const yplatform::ptree& conf) : server(settings(conf))
{
}

server::server(const settings& settings) : settings_(settings)
{
    auto webserver = yplatform::find<ymod_webserver::server>(settings_.web_server);
    auto front = settings_.front_endpoint;

    using ymod_webserver::transformer;
    using ymod_webserver::argument;
    using ymod_webserver::optional_argument;
    using ymod_webserver::default_validator;

    auto type_converter = [](const string& value) {
        ymod_xconf::config_type type;
        if (!ymod_xconf::resolve_type(value, type))
        {
            throw std::invalid_argument("unknown type");
        }
        return type;
    };
    webserver->bind(front, { "/ping" }, [](auto&& stream) {
        stream->result(ymod_webserver::codes::ok, "pong");
    });
    webserver->bind(
        front,
        { "/put" },
        check_tvm(
            settings_.tvm,
            boost::bind(&server::put_configuration, this, _1, _2, _3, _4, _5, _6, _7)),
        transformer(
            argument<string>("name"),
            argument<string>("owner"),
            argument<ymod_xconf::config_type>("type", default_validator<string>(), type_converter),
            optional_argument<string>("environment", ""),
            optional_argument<string>("token", ""),
            optional_argument<ymod_xconf::revision_t>("revision", ymod_xconf::INITIAL_REVISION)));
    webserver->bind(
        front,
        { "/list" },
        check_tvm(settings_.tvm, boost::bind(&server::list_configurations, this, _1, _2, _3, _4)),
        transformer(
            argument<ymod_xconf::config_type>("type", default_validator<string>(), type_converter),
            optional_argument<string>("environment", ""),
            optional_argument<ymod_xconf::revision_t>("revision", ymod_xconf::INITIAL_REVISION)));

    confdb_ = std::make_shared<ymod_xconf::pg_client>(
        yplatform::find<ymod_pq::call>("confdb"), settings_.allowed_environment);
}

void server::fini(void)
{
    confdb_.reset();
}

void server::put_configuration(
    ymod_webserver::http::stream_ptr stream,
    const string& name,
    const string& owner,
    ymod_xconf::config_type type,
    const string& env,
    const string& token,
    ymod_xconf::revision_t revision)
{
    using namespace ymod_webserver;
    namespace p = std::placeholders;

    auto data = boost::make_shared<string>(
        stream->request()->raw_body.begin(), stream->request()->raw_body.end());
    if (env.size())
    {
        auto environment = ymod_xconf::resolve_environment(env);
        if (environment == ymod_xconf::config_environment::UNKNOWN)
        {
            stream->result(codes::bad_request, "unknown environemnt");
            return;
        }
        confdb_->put(
            type,
            environment,
            name,
            owner,
            token,
            revision,
            data,
            std::bind(&server::handle_put, shared_from(this), stream, p::_1, p::_2),
            stream->ctx());
    }
    else
    {
        confdb_->put(
            type,
            name,
            owner,
            token,
            revision,
            data,
            std::bind(&server::handle_put, shared_from(this), stream, p::_1, p::_2),
            stream->ctx());
    }
}

void server::list_configurations(
    ymod_webserver::http::stream_ptr stream,
    ymod_xconf::config_type type,
    const string& env,
    ymod_xconf::revision_t revision)
{
    using namespace ymod_webserver;
    namespace p = std::placeholders;

    if (env.size())
    {
        auto environment = ymod_xconf::resolve_environment(env);
        if (environment == ymod_xconf::config_environment::UNKNOWN)
        {
            stream->result(codes::bad_request, "unknown environemnt");
            return;
        }
        confdb_->list(
            type,
            environment,
            revision,
            std::bind(&server::handle_list, shared_from(this), stream, p::_1, p::_2),
            stream->ctx());
    }
    else
    {
        confdb_->list(
            type,
            revision,
            std::bind(&server::handle_list, shared_from(this), stream, p::_1, p::_2),
            stream->ctx());
    }
}

void server::handle_put(
    ymod_webserver::http::stream_ptr stream,
    const ymod_xconf::error_code& ec,
    ymod_xconf::revision_t revision)
{
    using namespace ymod_webserver;
    if (ec)
    {
        stream->result(
            (ec == ymod_xconf::error::transport_error ? codes::bad_gateway : codes::bad_request),
            ec.message());
    }
    else
    {
        stream->result(codes::ok, std::to_string(revision));
    }
}

void server::handle_list(
    ymod_webserver::http::stream_ptr stream,
    const ymod_xconf::error_code& ec,
    const ymod_xconf::conf_list_ptr& configurations)
{
    using namespace ymod_webserver;
    if (ec)
    {
        stream->result(
            (ec == ymod_xconf::error::transport_error ? codes::bad_gateway : codes::bad_request),
            ec.message());
    }
    else
    {
        stream->result(codes::ok, configurations->serialize_to_msgpack());
    }
}

}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::conf::server)
