#pragma once

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/hashed_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/unordered_map.hpp>

#include <yplatform/future/future.hpp>
#include <ymod_blackbox/auth.h>
#include <yxiva/core/user_info.h>
#include <yxiva/core/types.h>
#include <yxiva/core/auth/authorizer.h>

namespace yxiva {

using namespace boost::multi_index;

class caching_authorizer
    : public authorizer
    , public enable_shared_from_this<caching_authorizer>
{

    typedef boost::multi_index_container<
        user_info,
        indexed_by<
            sequenced<>,
            hashed_unique<member<user_info, user_id, &user_info::uid>, boost::hash<string>>,
            hashed_unique<
                member<user_info, service_user_id, &user_info::suid>,
                boost::hash<string>>>>
        ui_dict;

public:
    caching_authorizer(authorizer_ptr impl, std::size_t max_cache_size = 1000U)
        : impl_(impl), max_cache_size_(max_cache_size)
    {
    }

    user_info_future_t authenticate(
        task_context_ptr ctx,
        const address& user_address,
        const string& login)
    {
        user_info_promise_t prom;
        auto middle_res = impl_->authenticate(ctx, user_address, login);
        middle_res.add_callback(
            boost::bind(&caching_authorizer::handle_auth, shared_from_this(), prom, middle_res));
        return prom;
    }
    user_info_future_t authenticate(
        task_context_ptr ctx,
        const address& user_address,
        const service_user_id& suid)
    {
        user_info_promise_t prom;
        {
            scoped_lock lock(mutex_);
            auto cache_item = find_in_cache(suid);
            if (cache_item)
            {
                prom.set(*cache_item);
                return prom;
            }
        }
        auto middle_res = impl_->authenticate(ctx, user_address, suid);
        middle_res.add_callback(
            boost::bind(&caching_authorizer::handle_auth, shared_from_this(), prom, middle_res));
        return prom;
    }
    user_info_future_t authenticate(
        task_context_ptr ctx,
        const address& user_address,
        const user_id& uid)
    {
        user_info_promise_t prom;
        {
            scoped_lock lock(mutex_);
            auto cache_item = find_in_cache(uid);
            if (cache_item)
            {
                prom.set(*cache_item);
                return prom;
            }
        }
        auto middle_res = impl_->authenticate(ctx, user_address, uid);
        middle_res.add_callback(
            boost::bind(&caching_authorizer::handle_auth, shared_from_this(), prom, middle_res));
        return prom;
    }

private:
    void handle_auth(user_info_promise_t prom, user_info_future_t middle_future_result)
    {
        try
        {
            auto ui = middle_future_result.get();
            {
                scoped_lock lock(mutex_);
                add_to_cache(ui);
            }
            prom.set(ui);
        }
        catch (...)
        {
            prom.set_current_exception();
        }
    }

    void add_to_cache(user_info const& ui)
    {
        auto info_it = cache_.push_front(ui);
        if (!info_it.second)
        {
            /* already in cache */
            cache_.relocate(cache_.begin(), info_it.first);
        }

        if (cache_.size() > max_cache_size_)
        {
            drop_too_old_info();
        }
    }

    void drop_too_old_info()
    {
        cache_.pop_back();
    }

    boost::optional<user_info> find_in_cache(user_id const& uid) const
    {
        boost::optional<user_info> result;
        auto& uid_index = cache_.get<1>();
        auto it = uid_index.find(uid);
        if (it != uid_index.end())
        {
            result = *it;
        }
        return result;
    }

    boost::optional<user_info> find_in_cache(service_user_id const& suid) const
    {
        boost::optional<user_info> result;
        auto& suid_index = cache_.get<2>();
        auto index_it = suid_index.find(suid);
        if (index_it != suid_index.end())
        {
            result = *index_it;
        }
        return result;
    }

    mutex mutex_;
    authorizer_ptr impl_;
    ui_dict cache_;

    std::size_t max_cache_size_;
};

}
