#pragma once

#include "types.h"
#include <yxiva/core/packing.hpp>
#include <yplatform/net/byte_order.h>

namespace yxiva::binary_protocol {

enum type : uint8_t
{
    data = 1,         // data frame from client to rproxy backends
    proxy_status = 2, // xiva response if rproxy backend is anavailable
    notification = 3, // push notifications from any authorized sender
};

enum error_code : uint16_t
{
    success = 0,
    protocol_error = 1,
    backend_call_error = 2,
    internal_error = 3,
    corrupted_data_header = 4,
    backend_not_found = 5,
    service_unavailable = 6,
    too_many_requests = 7,
    frame_too_large = 8
};

inline const string& to_string(error_code code)
{
    static const std::vector<string> names = { "success",
                                               "protocol_error",
                                               "backend_call_error",
                                               "internal_error",
                                               "corrupted_data_header",
                                               "backend_not_found",
                                               "service_unavailable",
                                               "too_many_requests",
                                               "frame_too_large" };
    return names.at(code);
}

}

MSGPACK_ADD_ENUM(yxiva::binary_protocol::error_code);

namespace yxiva::binary_protocol {

struct data_header // just a header, not a frame
{
    uint8_t service_index = 0;
    uint32_t reqid = 0;
    string path;

    MSGPACK_DEFINE(service_index, reqid, path);
};

struct proxy_status_frame
{
    uint32_t reqid = 0;
    error_code error = error_code::internal_error;

    MSGPACK_DEFINE(reqid, error);
};

struct notification_header
{
    string uid;
    string service;
    string event;
    string transit_id;

    MSGPACK_DEFINE(uid, service, event, transit_id);
};

namespace detail {
template <typename T, typename B>
inline size_t pack_enum(T t, B* out)
{
    using underlying_type = std::underlying_type_t<T>;
    underlying_type encoded = yplatform::net::host_2_network(static_cast<underlying_type>(t));
    const B* pencoded = static_cast<uint8_t*>(static_cast<void*>(&encoded));
    std::copy(pencoded, pencoded + sizeof(underlying_type), out);
    return sizeof(underlying_type);
}

template <typename T>
inline void unpack_numeric(const uint8_t* data, [[maybe_unused]] size_t size, T& t, size_t& offset)
{
    assert(offset + sizeof(T) <= size);
    T encoded;
    uint8_t* pencoded = static_cast<uint8_t*>(static_cast<void*>(&encoded));
    std::copy(data + offset, data + offset + sizeof(T), pencoded);
    offset += sizeof(T);
    t = yplatform::net::network_2_host(encoded);
}

template <typename T>
inline void unpack_enum(const uint8_t* data, size_t size, T& t, size_t& offset)
{
    using underlying_type = std::underlying_type_t<T>;
    underlying_type decoded;
    unpack_numeric(data, size, decoded, offset);
    t = static_cast<T>(decoded);
}
}

inline type unpack_type(const uint8_t* data, size_t size)
{
    type ret;
    size_t offset = 0;
    detail::unpack_enum(data, size, ret, offset);
    return ret;
}

inline auto unpack_type(const string& data)
{
    assert(data.size());
    return unpack_type(reinterpret_cast<const uint8_t*>(&data[0]), data.size());
}

inline auto unpack_data_header(const string& data)
{
    assert(data.size());

    data_header ret;
    auto type = unpack_type(data);
    if (type != type::data)
    {
        return std::pair(false, ret);
    }

    try
    {
        ::yxiva::unpack(&data[1], data.size() - 1, ret);
    }
    catch (msgpack::unpack_error& /*e*/)
    {
        return std::pair(false, ret); // TODO pass error code
    }
    catch (std::bad_cast& /*e*/)
    {
        return std::pair(false, ret);
    }
    return std::pair(true, ret);
}

template <typename T>
inline auto pack(type type, T t)
{
    string ret;
    ret.resize(1);
    ret[0] = static_cast<char>(type);
    ::yxiva::pack(t, ret);
    return ret;
}

inline string pack(const proxy_status_frame& frame)
{
    return pack(type::proxy_status, frame);
}

inline string pack(const notification_header& hdr)
{
    return pack(type::notification, hdr);
}

}
