#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/operation_result.h>
#include <yplatform/util/sstream.h>
#include <msgpack.hpp>
#include <string>
#include <set>
#include <map>

namespace yxiva {
enum class app_environment
{
    automatic,
    production,
    development
};

inline std::pair<operation::result, app_environment> resolve_app_environment(const string& name)
{
    static const std::map<string, app_environment> APP_ENVIRONMENT_BY_NAME{
        { "auto", app_environment::automatic },
        { "production", app_environment::production },
        { "development", app_environment::development }
    };
    auto it = APP_ENVIRONMENT_BY_NAME.find(name);
    if (it == APP_ENVIRONMENT_BY_NAME.end()) return { "unknown environment " + name, {} };
    return { operation::success, it->second };
}

inline const string& get_environment_name(app_environment env)
{
    static const string UNKNOWN = "unknown";
    static const std::map<app_environment, string> APP_ENVIRONMENT_NAMES{
        { app_environment::automatic, "auto" },
        { app_environment::production, "production" },
        { app_environment::development, "development" }
    };
    auto it = APP_ENVIRONMENT_NAMES.find(env);
    return it == APP_ENVIRONMENT_NAMES.end() ? UNKNOWN : it->second;
}

}

MSGPACK_ADD_ENUM(yxiva::app_environment);

namespace yxiva {
struct application_config
{
    string bb_client_id; // for bb 2-legged oauth authorization
    string xiva_service; // for xiva_token authorization

    string platform;
    string app_name;
    ttl_t ttl;

    string secret_key;
    // @todo: wns needs secret_id also
    int64_t expiration_time = 0;

    string key_backup;
    app_environment environment = app_environment::automatic;

    int64_t updated_at = 0;

    const string name() const
    {
        return platform + ":" + app_name;
    }
    const string owner_id() const
    {
        if (bb_client_id.size()) return "ID:" + bb_client_id;
        if (xiva_service.size()) return "xivaservice:" + xiva_service;
        assert(false && "owner_id is not determined");
        return {};
    }

    // WARNING: Do not remove existing fields, add new ones to the end
    MSGPACK_DEFINE(
        bb_client_id,
        xiva_service,
        platform,
        app_name,
        ttl,
        secret_key,
        expiration_time,
        key_backup,
        environment,
        updated_at);
};

// Base token_properties stores data common for all tokens.
// Token type specific data (limits and such) is to be stored in derived classes.
struct token_properties
{
    string service;
    string name;
    bool revoked;

    token_properties() : revoked(false)
    {
    }

    string item_name() const
    {
        string res;
        yplatform::sstream(res, service.size() + name.size() + 1) << service << ":" << name;
        return res;
    }

    // WARNING: Do not remove existing fields, add new ones to the end
    MSGPACK_DEFINE(service, name, revoked);
};

struct send_token_properties : public token_properties
{
    // WARNING: Do not remove existing fields, add new ones to the end
    MSGPACK_DEFINE(MSGPACK_BASE(token_properties));
};

// Token name is also used as xiva endpoint name.
struct listen_token_properties : public token_properties
{
    string client;

    // For legacy token support.
    std::set<string> allowed_services;
    bool is_listen_allowed(const string& name) const
    {
        return name == service || allowed_services.count(name);
    }

    // WARNING: Do not remove existing fields, add new ones to the end
    MSGPACK_DEFINE(MSGPACK_BASE(token_properties), client, allowed_services);
};

struct tvm_app_info
{
    uint32_t id = 0;
    string name;
    bool suspended = false;

    bool operator<(const tvm_app_info& other) const
    {
        return id < other.id;
    }

    bool operator==(const tvm_app_info& other) const
    {
        return id == other.id && name == other.name && suspended == other.suspended;
    }

    // WARNING: Do not remove existing fields, add new ones to the end
    MSGPACK_DEFINE(id, name, suspended);
};

struct service_properties
{
    string name;
    string owner_prefix;
    string owner_id;
    string description;
    bool is_passport;
    std::set<string> oauth_scopes;
    bool is_stream;
    uint16_t stream_count;
    bool revoked;
    bool auth_disabled;
    // This field is temporary while owner remains in xconf.
    string deprecated_owner; // xconf_owner_deprecated
    bool queued_delivery_by_default = true;
    // Key: environment name.
    std::map<string, std::set<tvm_app_info>> tvm_publishers;
    std::map<string, std::set<tvm_app_info>> tvm_subscribers;

    service_properties()
        : is_passport(false)
        , is_stream(false)
        , stream_count(0)
        , revoked(false)
        , auth_disabled(false)
    {
    }

    string owner() const
    {
        string res;
        yplatform::sstream(res, owner_prefix.size() + owner_id.size()) << owner_prefix << owner_id;
        return res;
    }

    // WARNING: Do not remove existing fields, add new ones to the end
    MSGPACK_DEFINE(
        name,
        owner_prefix,
        owner_id,
        description,
        is_passport,
        oauth_scopes,
        is_stream,
        stream_count,
        revoked,
        auth_disabled,
        queued_delivery_by_default,
        tvm_publishers,
        tvm_subscribers);
};

}
