#pragma once

#include <openssl/ec.h>
#include <openssl/evp.h>
#include <openssl/ecdsa.h>
#include <openssl/rand.h>
#include <memory>
#include <stdexcept>
#include <vector>

namespace yxiva { namespace ec_crypto {

// aliases and functions implementing RAII for OpenSSL entities
using evp_pkey_ptr = std::unique_ptr<EVP_PKEY, void (&)(EVP_PKEY*)>;
using evp_pkey_ctx_ptr = std::unique_ptr<EVP_PKEY_CTX, void (&)(EVP_PKEY_CTX*)>;
using evp_cipher_ctx_ptr = std::unique_ptr<EVP_CIPHER_CTX, void (&)(EVP_CIPHER_CTX*)>;
using evp_md_ctx_ptr = std::unique_ptr<EVP_MD_CTX, void (&)(EVP_MD_CTX*)>;
using ec_key_ptr = std::unique_ptr<EC_KEY, void (&)(EC_KEY*)>;
using ec_point_ptr = std::unique_ptr<EC_POINT, void (&)(EC_POINT*)>;
using ecdsa_sig_ptr = std::unique_ptr<ECDSA_SIG, void (&)(ECDSA_SIG*)>;
using ec_group_ptr = std::unique_ptr<EC_GROUP, void (&)(EC_GROUP*)>;
using bio_ptr = std::unique_ptr<BIO, int (&)(BIO*)>;

inline evp_pkey_ptr make_ptr(EVP_PKEY* k)
{
    return evp_pkey_ptr(k, EVP_PKEY_free);
}
inline evp_pkey_ctx_ptr make_ptr(EVP_PKEY_CTX* c)
{
    return evp_pkey_ctx_ptr(c, EVP_PKEY_CTX_free);
}
inline evp_cipher_ctx_ptr make_ptr(EVP_CIPHER_CTX* c)
{
    return evp_cipher_ctx_ptr(c, EVP_CIPHER_CTX_free);
}
inline evp_md_ctx_ptr make_ptr(EVP_MD_CTX* c)
{
    return evp_md_ctx_ptr(c, EVP_MD_CTX_free);
}
inline ec_key_ptr make_ptr(EC_KEY* k)
{
    return ec_key_ptr(k, EC_KEY_free);
}
inline ec_point_ptr make_ptr(EC_POINT* p)
{
    return ec_point_ptr(p, EC_POINT_free);
}
inline ecdsa_sig_ptr make_ptr(ECDSA_SIG* s)
{
    return ecdsa_sig_ptr(s, ECDSA_SIG_free);
}
inline ec_group_ptr make_ptr(EC_GROUP* g)
{
    return ec_group_ptr(g, EC_GROUP_free);
}
inline bio_ptr make_ptr(BIO* bio)
{
    return bio_ptr(bio, BIO_free);
}

// converts key from it's evp form to ec
inline ec_key_ptr evp_to_ec(const evp_pkey_ptr& key)
{
    auto ec_key = make_ptr(EVP_PKEY_get1_EC_KEY(key.get()));
    if (!ec_key) throw std::runtime_error("evp to ec conversion failed");
    return ec_key;
}

// generates cryptographycally "good" random vector
inline std::vector<unsigned char> rand_vector(unsigned int size)
{
    std::vector<unsigned char> vec(size);
    if (size && !RAND_bytes(vec.data(), static_cast<int>(vec.size())))
        throw std::runtime_error("random vector generation error");
    return vec;
}

// generates elliptic keys on "prime256v1" named curve
evp_pkey_ptr generate_crypto_keys();

// hkdf cryptographic function for key derivation
std::vector<unsigned char> hkdf(
    const std::vector<unsigned char>& salt,
    const std::vector<unsigned char>& key,
    const std::vector<unsigned char>& info,
    size_t length);

// gets public key in binary form from elliptic key
std::vector<unsigned char> get_public_key(const ec_key_ptr& key);

// reads binary elliptic public key into evp format key
evp_pkey_ptr evp_from_public_key(const std::vector<unsigned char>& pk, const EC_GROUP* ecgrp);

// derives shared secret from user and server elliptic keys
std::vector<unsigned char> derive_secret(
    const evp_pkey_ptr& server_key,
    const evp_pkey_ptr& client_key);

// aesgcm encryption with padding (only 0 currently supported), used for webpush payload encryption
std::string aes_gcm_128_with_padding(
    const std::string& data,
    const std::vector<unsigned char>& key,
    const std::vector<unsigned char>& iv,
    unsigned short padding = 0);

// signs data with elliptic key and sha256 digest, used for vapid signing
std::vector<unsigned char> sign(const void* data, size_t len, const evp_pkey_ptr& key);

// verify data signed with elliptic key and sha256 digest
bool verify_sign(
    const void* data,
    size_t len,
    const std::vector<unsigned char>& sig_raw,
    const evp_pkey_ptr& key);

// reads private key from pem file, used for vapid signing
evp_pkey_ptr read_pem(const std::string& filename);

evp_pkey_ptr read_pem_buf(const std::string& buf);

}}
