#pragma once

#include <yxiva/core/packing.hpp>
#include <vector>

namespace yxiva {
struct message;
struct sub_t;

namespace filter {

enum class message_condition_type
{
    data_field_equals,
    has_tags,
    has_least_one,
    event,
    unknown
};

enum class subscription_condition_type
{
    platform,
    transport,
    subscription_id,
    session,
    uuid,
    device,
    app,
    unknown
};

}
}

MSGPACK_ADD_ENUM(yxiva::filter::message_condition_type);
MSGPACK_ADD_ENUM(yxiva::filter::subscription_condition_type);

namespace yxiva {
namespace filter {

template <typename ConditionType>
bool condition_uses_key(const ConditionType type);

template <typename ConditionType = message_condition_type>
ConditionType condition_from_string(std::string const& str);

template <typename ConditionType = message_condition_type>
const char* condition_to_string(const ConditionType condition);

enum class action
{
    send_bright,
    send_silent,
    skip,
    unknown
};

action action_from_string(std::string const& str);
const char* action_to_string(const action action);

template <typename MatchedType, typename ConditionType>
struct basic_condition
{
    using matched_t = MatchedType;
    using cond_t = ConditionType;

    cond_t type;
    std::string key;
    std::vector<std::string> value;

    bool matches(const matched_t& message) const;

    bool operator==(const basic_condition& other) const
    {
        return type == other.type && key == other.key && value == other.value;
    }

    MSGPACK_DEFINE(type, key, value);
};

using condition = basic_condition<message, message_condition_type>;
using subscription_condition = basic_condition<sub_t, subscription_condition_type>;

template <typename Condition>
using basic_var_t = std::pair<string, Condition>;
template <typename Condition>
using basic_vars_t = std::map<string, Condition>;

using var_t = basic_var_t<condition>;
using vars_t = basic_vars_t<condition>;

enum operator_t
{
    OP_AND,
    OP_NOT
};

struct expression
{
    std::vector<operator_t> operators;
    std::vector<string> operands;

    bool empty() const
    {
        return operands.empty();
    }

    bool operator==(const expression& other) const;
};

struct rule
{
    yxiva::filter::action action;
    expression expr;

    bool matches(
        const message& msg,
        std::map<std::string, bool>& cached_vars,
        const vars_t& definitions) const;
    bool operator==(const rule& other) const;
};

typedef std::vector<filter::rule> rules_t;

bool is_fake_variable(const string& var_name);
void add_single_variable_rule(rules_t& rules, vars_t& vars, condition&& cond, action act);

}

class filter_set
{
public:
    filter_set();

    filter_set(filter::rules_t&& rules, filter::vars_t&& vars, filter::action default_action);

    filter::action apply(const message& message) const;

    const std::string& to_string();

    bool operator==(const filter_set& other) const;

    // Getters for test purposes only
    const filter::rules_t& get_rules() const
    {
        return rules_;
    }
    const filter::vars_t& get_vars() const
    {
        return vars_;
    }
    const filter::action& get_default() const
    {
        return default_action_;
    }

private:
    string json_str_;

    filter::rules_t rules_;
    filter::vars_t vars_;
    filter::action default_action_;
};

}
