#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/methods/check.h> // validate_param
#include <boost/crc.hpp>
#include <string>

namespace yxiva {

typedef uint32_t gid_t;
static constexpr unsigned NUM_GIDS = 65536;
static constexpr unsigned MAX_GID = NUM_GIDS - 1;

namespace detail {
inline bool try_strtoull(const char* begin, const char* end, unsigned long long& output)
{
    bool success = false;
    if (begin != end && validate_param(begin, end, isdigit))
    {
        char* pos = nullptr;
        auto number = std::strtoull(begin, &pos, 10);
        success = (pos != begin && pos == end && errno != ERANGE);
        errno = 0;
        if (success) output = number;
    }
    return success;
}

inline bool try_strtoull(const std::string& s, unsigned long long& output)
{
    return try_strtoull(s.data(), s.data() + s.size(), output);
}

inline gid_t gid_from_uid(const char* begin, const char* end)
{
    gid_t gid;
    unsigned long long number;
    if (detail::try_strtoull(begin, end, number))
    {
        gid = number % NUM_GIDS;
    }
    else
    {
        boost::crc_16_type result;
        result.process_block(begin, end);
        auto checksum = result.checksum();
        static_assert(
            std::is_same<decltype(checksum), uint16_t>::value,
            "The checksum must be a uint16 value");
        gid = checksum;
    }
    return gid;
}
}

static constexpr char UID_TAIL_DELIMITER = '+';

inline gid_t gid_from_uid(const std::string& uid)
{
    auto tail_offset = uid.find(UID_TAIL_DELIMITER);
    const char* begin = uid.data();
    const char* end = begin + (tail_offset == string::npos ? uid.size() : tail_offset);

    return detail::gid_from_uid(begin, end);
}

}
