#pragma once

#include <yxiva/core/subscriptions.h>
#include <yxiva/core/types.h>
#include <yxiva/core/time_traits.h>
#include <yxiva/core/filter/filter_set.h>
#include <msgpack/type.hpp>
#include <algorithm>

namespace yxiva {

enum class message_content_type : uint8_t
{
    unknown = 0,
    json = 1,
    binary = 2
};

enum class message_flags : uint32_t
{
    none = 0,
    // applying filter set always returns send_bright
    ignore_filters = 1 << 0
};

message_flags operator|(message_flags flag1, message_flags flag2);
message_flags& operator|=(message_flags& flag1, message_flags flag2);
}

MSGPACK_ADD_ENUM(yxiva::message_content_type);
MSGPACK_ADD_ENUM(yxiva::message_flags);

namespace yxiva {
/// WARN
/// DO NOT REMOVE OR MODIFY ITEMS, ONLY PUSH BACK
struct message
{
    string uid;
    string suid; // unused
    string service;

    string operation; // API event
    string lcn;
    string session_key;

    std::map<string, string> data; // API keys
    string raw_data;               // API payload

    bool bright;

    string transit_id;

    uint64_t local_id;

    int64_t event_ts;

    std::vector<std::string> tags;

    std::map<std::string, std::string> repacking_rules;

    ttl_t ttl; // seconds

    message_flags flags;

    string topic;

    int64_t event_ts_ms;

    std::vector<filter::subscription_condition> subscription_matchers;

    std::set<string> experiments;

    message_content_type type = message_content_type::unknown;

    uint32_t batch_size = 0; // non-zero for batch notifications

    bool broadcast = false;

    message()
        : bright(true)
        , local_id(0)
        , event_ts(0)
        , ttl(86400)
        , flags(message_flags::none)
        , event_ts_ms(0)
    {
    }

    // WARN Do not forget to update packet::msgpack_pack().
    MSGPACK_DEFINE(
        uid,
        suid,
        service,
        operation,
        lcn,
        session_key,
        data,
        raw_data,
        bright,
        transit_id,
        local_id,
        event_ts,
        tags,
        repacking_rules,
        ttl,
        flags,
        topic,
        event_ts_ms,
        subscription_matchers,
        experiments,
        type,
        batch_size,
        broadcast);

    bool has_tag(const std::string& tag_to_find) const;
    bool has_flag(message_flags flag) const;
    void set_flag(message_flags flag);
    void unset_flag(message_flags flag);
};

inline ttl_t effective_ttl(const ttl_t ttl)
{
    // ttl values less than the following are treated as zero
    static constexpr ttl_t MINIMUM_EFFECTIVE_TTL = 5;
    return ttl < MINIMUM_EFFECTIVE_TTL ? 0 : ttl;
}

inline ttl_t remaining_ttl(const message& msg, time_t now = std::time(nullptr))
{
    auto remaining = std::min<ttl_t>(msg.event_ts - now + msg.ttl, msg.ttl);
    return remaining > 0 ? effective_ttl(static_cast<ttl_t>(remaining)) : 0;
}

inline time_ms_t elapsed_time_ms(const message& msg)
{
    // Messages with zero event_ts_ms are also
    // supported for backward compatibility.
    if (msg.event_ts_ms)
    {
        auto res = time_ms();
        return res > msg.event_ts_ms ? res - msg.event_ts_ms : 0;
    }

    auto res = std::time(nullptr);
    return res > msg.event_ts ? (res - msg.event_ts) * 1000 : 0;
}

/// WARN
/// DO NOT REMOVE OR MODIFY ITEMS, ONLY PUSH BACK
struct packet
{
    task_context_ptr ctx;
    const string& uid;
    const string& service;
    const ::yxiva::message& message;
    const sub_t& subscription;
    bool bright;
    ttl_t ttl;
    bool robust_delivery;

    packet(
        task_context_ptr ctx,
        const ::yxiva::message& message,
        const sub_t& subscription,
        time_t now = std::time(nullptr))
        : ctx(ctx)
        , uid(message.uid.empty() ? subscription.uid : message.uid)
        , service(message.service.empty() ? subscription.service : message.service)
        , message(message)
        , subscription(subscription)
        , bright(message.bright)
        , ttl(remaining_ttl(message, now))
        , robust_delivery(false)
    {
    }

    packet& mark_queued_delivery()
    {
        robust_delivery = true;
        return *this;
    }

    packet& mark_silent_delivery()
    {
        bright = false;
        return *this;
    }

    // WARN
    // Keep identical to message::MSGPACK_DEFINE.
    template <typename Packer>
    void msgpack_pack(Packer& pk) const
    {
        msgpack::type::make_define_array(
            uid,
            message.suid,
            service,
            message.operation,
            message.lcn,
            message.session_key,
            message.data,
            message.raw_data,
            bright,
            message.transit_id,
            message.local_id,
            message.event_ts,
            message.tags,
            message.repacking_rules,
            ttl,
            message.flags,
            message.topic,
            message.event_ts_ms,
            message.subscription_matchers,
            message.experiments,
            message.type,
            message.batch_size,
            message.broadcast)
            .msgpack_pack(pk);
    }
};

typedef boost::shared_ptr<message> message_ptr;

}
