#pragma once

#include <yplatform/log.h>
#include <ymod_webserver/codes.h>
#include <ymod_webserver/request.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/expirable_stream.h>

#include "request_adaptor.h"

namespace {

inline bool isalnum_list(char c)
{
    return isalnum(c) || c == ',';
}

inline bool isalnum_ext(char c)
{
    return isalnum(c) || c == '-' || c == '_' || c == '.';
}

inline bool isalnum_ext_list(char c)
{
    return isalnum_ext(c) || c == ',';
}

inline bool tag_char(char c)
{
    return isalnum(c) || c == '.' || c == '_';
}

// Extended tag charset for use in filters.
// The "-" char is not allowed in short service+tags notation.
inline bool tag_char_ex(char c)
{
    return tag_char(c) || c == '-';
}

inline bool tag_list_char(char c)
{
    return tag_char(c) || c == ',';
}

template <typename Iterator, typename CheckFunc>
inline bool validate_param(Iterator begin, Iterator end, const CheckFunc& func)
{
    return std::find_if_not(begin, end, func) == end;
}

template <typename CheckFunc>
inline bool validate_param(const std::string& param, const CheckFunc& func)
{
    return validate_param(param.begin(), param.end(), func);
}

template <typename Response>
inline void result(Response resp, ymod_webserver::codes::code cd, const std::string& message)
{
    resp->detach_stream()->result(cd, message);
}

template <>
inline void result(
    ymod_webserver::http::stream_ptr resp,
    ymod_webserver::codes::code cd,
    const std::string& message)
{
    resp->result(cd, message);
}

template <>
inline void result(
    ymod_webserver::http::expirable_stream_ptr resp,
    ymod_webserver::codes::code cd,
    const std::string& message)
{
    resp->result(cd, message);
}

#define BAD_REQUEST(resp, req, message)                                                            \
    {                                                                                              \
        string message_tmp_str = message;                                                          \
        YLOG_CTX_GLOBAL(req->ctx(), info) << "bad request: " << message_tmp_str;                   \
        result(resp, ymod_webserver::codes::bad_request, message_tmp_str);                         \
        return true;                                                                               \
    }

#define FIND_PARAM(resp, req, a)                                                                   \
    string a = req->url.param_value(#a, "");                                                       \
    if (a.empty()) BAD_REQUEST(resp, req, string("no ") + #a + " specified")

#define FIND_OPTIONAL_PARAM(req, a) string a = req->url.param_value(#a, "");

#define VALIDATE_ALNUM(resp, a)                                                                    \
    if (std::find_if_not(a.begin(), a.end(), isalnum) != a.end())                                  \
    BAD_REQUEST(resp, resp->request(), string("bad characters in ") + #a + " parameter")

#define VALIDATE_ALNUM_EXT(resp, a)                                                                \
    if (std::find_if_not(a.begin(), a.end(), isalnum_ext) != a.end())                              \
    BAD_REQUEST(resp, resp->request(), string("bad characters in ") + #a + " parameter")

#define VALIDATE_ALNUM_EXT_LIST(resp, a)                                                           \
    if (std::find_if_not(a.begin(), a.end(), isalnum_ext_list) != a.end())                         \
    BAD_REQUEST(resp, resp->request(), string("bad characters in ") + #a + " parameter")

#define VALIDATE_NUM(resp, a)                                                                      \
    if (std::find_if_not(a.begin(), a.end(), isdigit) != a.end())                                  \
    BAD_REQUEST(resp, resp->request(), string("not digits in ") + #a + " parameter")

#define VALIDATE_CALLBACK(resp, a)                                                                 \
    {                                                                                              \
    }

#define VALIDATE_EX(resp, func, a)                                                                 \
    if (!validate_param(a, func))                                                                  \
    BAD_REQUEST(resp, resp->request(), string("bad characters in ") + #a + " parameter")

#define FIND_PARAM_EX(resp, req, func, a)                                                          \
    FIND_PARAM(resp, req, a)                                                                       \
    VALIDATE_EX(resp, func, a)

#define FIND_OPTIONAL_PARAM_EX(resp, req, func, a)                                                 \
    FIND_OPTIONAL_PARAM(req, a)                                                                    \
    VALIDATE_EX(resp, func, a)

#define FIND_CALLBACK_PARAM(resp, req, a)                                                          \
    FIND_PARAM(resp, req, a)                                                                       \
    VALIDATE_CALLBACK(resp, a)

#define FIND_ALNUM_PARAM(resp, req, a)                                                             \
    FIND_PARAM(resp, req, a)                                                                       \
    VALIDATE_ALNUM(resp, a)

#define FIND_ALNUM_EXT_PARAM(resp, req, a)                                                         \
    FIND_PARAM(resp, req, a)                                                                       \
    VALIDATE_ALNUM_EXT(resp, a)

#define FIND_ALNUM_EXT_LIST_PARAM(resp, req, a)                                                    \
    FIND_PARAM(resp, req, a)                                                                       \
    VALIDATE_ALNUM_EXT_LIST(resp, a)

#define FIND_NUM_PARAM(resp, req, a)                                                               \
    FIND_PARAM(resp, req, a)                                                                       \
    VALIDATE_NUM(resp, a)

#define FIND_OPTIONAL_ALNUM_PARAM(resp, req, a)                                                    \
    FIND_OPTIONAL_PARAM(req, a)                                                                    \
    VALIDATE_ALNUM(resp, a)

#define FIND_OPTIONAL_ALNUM_EXT_PARAM(resp, req, a)                                                \
    FIND_OPTIONAL_PARAM(req, a)                                                                    \
    VALIDATE_ALNUM_EXT(resp, a)

#define FIND_OPTIONAL_ALNUM_EXT_LIST_PARAM(resp, req, a)                                           \
    FIND_OPTIONAL_PARAM(req, a)                                                                    \
    VALIDATE_ALNUM_EXT_LIST(resp, a)

#define FIND_OPTIONAL_NUM_PARAM(resp, req, a)                                                      \
    FIND_OPTIONAL_PARAM(req, a)                                                                    \
    VALIDATE_NUM(resp, a)

template <typename S>
inline bool fill_parameter_from_req(std::string& dest, const std::string& name, S s)
{
    dest = s->request()->url.param_value(name, "");
    if (dest.empty())
    {
        s->result(ymod_webserver::codes::bad_request, std::string("no ") + name + " specified");
        return false;
    }
    return true;
}

template <>
inline bool fill_parameter_from_req(
    std::string& dest,
    const std::string& name,
    const ::yxiva::request_adaptor_ptr& req)
{
    dest = req->native_request()->url.param_value(name, "");
    if (dest.empty())
    {
        req->send_error(
            ymod_webserver::codes::bad_request, std::string("no ") + name + " specified");
        return false;
    }
    return true;
}

}
