#pragma once

#include <boost/algorithm/string/case_conv.hpp>
#include <yplatform/log.h>

#include <yxiva/core/types.h>
#include <yxiva/core/methods/request_adaptor.h>
#include <yxiva/core/methods/key_extractor.h>

namespace yxiva { namespace methods {

template <class Request, class... Args>
class dispatcher : public yplatform::log::contains_logger
{
public:
    typedef boost::function<void(Request req, Args...)> method_t;
    typedef std::map<string, method_t> methods_map_t;

    typedef boost::function<string(Request req)> key_extract_t;

public:
    dispatcher(const key_extract_t& extractor, const methods_map_t& methods = methods_map_t())
        : key_extractor_(extractor), methods_(methods)
    {
    }

    method_t& operator[](const string& key)
    {
        return methods_[key];
    }
    method_t& operator[](const char* key)
    {
        return methods_[key];
    }

    const method_t& operator[](const string& key) const
    {
        return methods_[key];
    }
    const method_t& operator[](const char* key) const
    {
        return methods_[key];
    }

    void operator()(Request req, Args... args) noexcept
    {
        try
        {
            string key = key_extractor_(req);
            boost::algorithm::to_lower(key);

            auto method = methods_.find(key);
            if (method != methods_.end())
            {
                method->second(req, args...);
                return;
            }
            send_not_found(req, args...);
        }
        catch (const std::exception& ex)
        {
            YLOG_CTX_LOCAL(req->ctx(), error) << "error while dispatching request: " << ex.what();
        }
        catch (...)
        {
            YLOG_CTX_LOCAL(req->ctx(), error) << "unexpected error while dispatching request";
        }
    }

private:
    template <class... CallArgs>
    void send_not_found(request_adaptor_ptr req, CallArgs...)
    {
        req->send_error(ymod_webserver::codes::not_found, "The requested URL was not found");
    }

    template <class... CallArgs>
    void send_not_found(
        ymod_webserver::request_ptr /*req*/,
        ymod_webserver::response_ptr resp,
        CallArgs... args)
    {
        send_not_found(resp, args...);
    }

    template <class... CallArgs>
    void send_not_found(ymod_webserver::response_ptr resp, CallArgs...)
    {
        resp->result(ymod_webserver::codes::not_found, "The requested URL was not found");
    }

private:
    key_extract_t key_extractor_;
    methods_map_t methods_;
};

}}
