#pragma once

#include "platforms/apns.h"
#include "platforms/apns_queue.h"
#include "platforms/fcm.h"
#include "platforms/hms.h"
#include "platforms/wns.h"
#include <yxiva/core/types.h>
#include <boost/algorithm/string.hpp>
#include <string>
#include <set>
#include <map>
#include <algorithm>

namespace yxiva {
namespace platform {

static const std::string MPNS = "mpns";
static const std::string COURIER = "courier";

static const std::set<std::string> ALL_SUPPORTED = { APNS, FCM, WNS, MPNS, APNS_QUEUE, HMS };

static const std::set<std::string> ALL_UNSUPPORTED = { COURIER };

struct resolved
{
    const std::string& name;
    bool supported;
};

inline resolved resolve_alias(const std::string& alias)
{
    auto supported_it = ALL_SUPPORTED.find(alias);
    if (supported_it != ALL_SUPPORTED.end()) return { *supported_it, true };
    auto unsupported_it = ALL_UNSUPPORTED.find(alias);
    if (unsupported_it != ALL_UNSUPPORTED.end()) return { *unsupported_it, false };

    static const std::map<std::string, std::string> aliases = {
        { "a", FCM },   { "android", FCM }, { "gcm", FCM }, // gcm_compatibility
        { "fcm", FCM }, { "i", APNS },      { "ios", APNS }, { "apns_queue", APNS_QUEUE },
    };
    static const std::string UNKNOWN;
    auto ifixed_name = aliases.find(alias);
    if (ifixed_name != aliases.end()) return { ifixed_name->second, true };
    return { UNKNOWN, false };
}

// Some platforms (currently only APNS_QUEUE) are virtual, providing additional capabilities
// to the real platform. For some applications, real platform's value is needed.
inline const std::string& get_real(const std::string& platform)
{
    if (platform == APNS_QUEUE) return APNS;
    return platform;
}

} // platform

static const string FCM_TOKEN_BLACKLISTED = "BLACKLISTED";

inline bool is_usable_token(const string& token, const string& platform)
{
    if (token.empty()) return false;
    if (platform == platform::FCM && boost::iequals(token, FCM_TOKEN_BLACKLISTED)) return false;
    return true;
}

inline bool android_platform(const string& platform)
{
    return platform == platform::FCM || platform == platform::HMS;
}

}
