#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/json.h>
#include <yxiva/core/message.h>
#include <yxiva/core/subscriptions.h>
#include <yxiva/core/operation_result.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/platforms.h>
#include <yplatform/encoding/url_encode.h>

namespace yxiva {

class push_request
{
    string payload_;
    string http_url_params_;
    bool bright_{ false };

public:
    push_request()
    {
    }

    push_request(string payload, bool bright = false)
        : payload_(std::move(payload)), bright_(bright)
    {
    }

    void add_push_service_parameter(const string& name, const string& value)
    {
        http_url_params_ += "&";
        http_url_params_ += yplatform::url_encode("x-" + name);
        http_url_params_ += "=";
        http_url_params_ += yplatform::url_encode(value);
    }

    void add_custom_parameter(const string& name, const string& value)
    {
        http_url_params_ += "&";
        http_url_params_ += yplatform::url_encode(name);
        http_url_params_ += "=";
        http_url_params_ += yplatform::url_encode(value);
    }

    void set_payload(string payload)
    {
        payload_ = std::move(payload);
    }

    void set_bright(bool bright = true)
    {
        bright_ = bright;
    }

    bool is_bright() const
    {
        return bright_;
    }

    const string& http_url_params() const
    {
        return http_url_params_;
    }

    const string& payload() const
    {
        return payload_;
    }

    std::size_t size() const
    {
        return payload_.size() + http_url_params_.size();
    }
};

class push_requests_queue
{
public:
    push_requests_queue() : requests(std::make_shared<std::vector<push_request>>())
    {
    }

    template <typename... Args>
    void emplace_back(Args&&... args)
    {
        requests->emplace_back(std::forward<Args>(args)...);
        bright = bright || requests->back().is_bright();
    }

    void push_back(push_request msg)
    {
        requests->emplace_back(std::move(msg));
        bright = bright || requests->back().is_bright();
    }

    const push_request& front() const
    {
        assert(next < requests->size());
        return requests->at(next);
    }

    const push_request& pop_front()
    {
        assert(next < requests->size());
        ++next;
        return requests->at(next - 1);
    }

    std::size_t size() const
    {
        return requests->size() - next;
    }

    bool empty() const
    {
        return size() == 0;
    }

    bool is_bright() const
    {
        return bright;
    }

    std::size_t payload_size() const
    {
        return requests->empty() ? 0 : requests->at(0).payload().size();
    }

private:
    std::shared_ptr<std::vector<push_request>> requests;
    std::size_t next = 0;
    bool bright{ false };
};

struct repack_features
{
    bool inject_transit_id = false;
    bool transit_id_appmetrica_format = false;
    bool inject_collapse_id = false;
    bool inject_uts = false;

    bool operator!=(const repack_features& other) const
    {
        return this != &other &&
            std::tie(
                inject_transit_id, transit_id_appmetrica_format, inject_collapse_id, inject_uts) !=
            std::tie(
                other.inject_transit_id,
                other.transit_id_appmetrica_format,
                other.inject_collapse_id,
                other.inject_uts);
    }
};

struct push_subscription_params : sub_t
{
    string app_name;
    string push_token;
    struct repack_features repack_features;

    push_subscription_params()
    {
    }

    push_subscription_params(const sub_t& base) : sub_t(base)
    {
        callback_uri::parse_mobile_uri(callback_url, app_name, push_token);
    }
};

struct apns_queue_sub_t : push_subscription_params
{
    string queue_id;

    apns_queue_sub_t(const sub_t& base)
    {
        static_cast<sub_t&>(*this) = base;
        // Device and service should already be stored in the base subscription,
        // but are retrieved from callback just in case.
        callback_uri::parse_apns_queue_uri(callback_url, service, app_name, device);
        queue_id = apns_queue_id(device, app_name);
    }
};

class push_request_cache
{
    // key is mobile platform name, app name and bright
    // (add everything affecting repack)
    using key_t = std::tuple<string, string, bool>;
    using value_t = std::pair<operation::result, push_requests_queue>;
    using storage_t = std::map<key_t, value_t>;

public:
    void store(
        const push_subscription_params& sub,
        const packet& packet,
        const operation::result& res,
        const push_requests_queue& queue)
    {
        static const push_requests_queue empty;
        storage_.emplace(
            std::piecewise_construct,
            std::tie(sub.platform, sub.app_name, packet.bright),
            std::tie(res.error_reason, res ? queue : empty));
    }

    boost::optional<const value_t&> try_get(
        const push_subscription_params& sub,
        const packet& packet)
    {
        auto it = storage_.find(std::tie(sub.platform, sub.app_name, packet.bright));
        if (it == storage_.end())
        {
            return boost::none;
        }
        else
        {
            return it->second;
        }
    }

private:
    storage_t storage_;
};

using push_request_cache_ptr = std::shared_ptr<push_request_cache>;

class field
{
public:
    string source_key;
    string target_key;

    field(const string& _target_key) : source_key(_target_key), target_key(_target_key)
    {
    }

    field(string _target_key, string _source_key)
        : source_key(std::move(_source_key)), target_key(std::move(_target_key))
    {
    }
};

inline void transform(
    json_value_ref& dst,
    const json_value_ref& src,
    const std::vector<field>& fields)
{
    for (auto& field : fields)
    {
        auto value = src[field.source_key];
        if (value)
        {
            dst[field.target_key] = std::move(value);
        }
        // @todo check field is optional and/or return error
    }
}

class repacker
{
public:
    virtual ~repacker() = default;
    virtual std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) = 0; // ? TODO
};

using repacker_ptr = std::shared_ptr<repacker>;

class custom_repacker : public repacker
{
public:
    static operation::result from_json_string(
        const string& raw_repacking_rules,
        custom_repacker& custom_repacker);

    static operation::result from_json(
        json_value js_repacking_rules,
        custom_repacker& custom_repacker);

    operation::result validate(const json_value& source_payload) const;

    std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) override;

private:
    bool pass_payload_as_is = false;
    bool include_source_payload = false;
    std::vector<field> payload;
    json_value push_service_params;
};

class apns_queue_repacker : public repacker
{
public:
    std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) override;
};

class bypass_repacker : public repacker
{
public:
    std::tuple<operation::error, push_requests_queue> repack(
        const packet& packet,
        const push_subscription_params& subscription) override;
};

class webpush_repacker
{
public:
    string repack(const packet& packet, const repack_features& features);
};

operation::result repack_message_if_needed(
    const packet& packet,
    const push_subscription_params& subscription,
    push_requests_queue& requests);

operation::result repack_message_if_needed(
    const packet& packet,
    const push_subscription_params& subscription,
    push_requests_queue& requests,
    push_request_cache& cache);

}