#pragma once

#include <yxiva/core/resharding/migration.h>
#include <yxiva/core/types.h>
#include <ymod_pq/response_handler.h>
#include <boost/lexical_cast.hpp>

namespace yxiva { namespace resharding {

class range_request_handler : public ymod_pq::response_handler
{
    migration current_{ migration::state_type::pending, 0, 0 };

    std::vector<migration> result_;
    string failure_reason_;
    bool bad_state_ = false;

    migration::state_type state_from_string(const string& s)
    {
        int i = boost::lexical_cast<int>(s);
        switch (i)
        {
        case 0:
            return migration::state_type::pending;
        case 1:
            return migration::state_type::ready;
        case 2:
            return migration::state_type::inprogress;
        case 3:
            return migration::state_type::finished;
        default:
            break;
        }
        throw std::runtime_error("unknown migration state " + s);
    }

public:
    enum
    {
        state,
        end_gid,

        column_cnt
    };

    virtual void handle_cell(unsigned /*row*/, unsigned col, const string& value, bool is_null)
        override
    {
        if (failure_reason_.size())
        {
            return;
        }

        if (is_null)
        {
            failure_reason_ = "migration parameter " + std::to_string(col) + " is null";
            return;
        }

        try
        {
            switch (col)
            {
            case state:
                current_.state = state_from_string(value);
                break;
            case end_gid:
                current_.end_gid = boost::lexical_cast<gid_t>(value);
                break;
            default:
                throw std::runtime_error("unknown pq column in range_request_handler!");
            }
        }
        catch (const std::exception& e)
        {
            failure_reason_ += "invalid field values (or number of fields) in pq response: ";
            failure_reason_ += e.what();
        }
    }

    void handle_row_end(unsigned /*row*/) override
    {
        // A tuple with end_gid equal to 65536
        // denotes that we tried an invalid
        // migration state transition.
        static const gid_t INVALID_MIGRATION_GID = 65536;
        if (current_.end_gid != INVALID_MIGRATION_GID)
        {
            result_.push_back(current_);
            current_.start_gid = current_.end_gid + 1;
            current_.end_gid = 0;
        }
        else
        {
            bad_state_ = true;
            current_ = migration();
        }
    }

    unsigned column_count() const override
    {
        return column_cnt;
    }

    const string& failure_reason() const
    {
        static const string empty_response = "db returned no migrations";
        return result_.empty() ? empty_response : failure_reason_;
    }

    const std::vector<migration>& data() const
    {
        return result_;
    }

    bool bad_state() const
    {
        return bad_state_;
    }

    std::vector<migration> move_data()
    {
        return std::move(result_);
    }
};

}}
