#pragma once

#include <yxiva/core/resharding/migration_storage.h>
#include <yxiva/core/resharding/pq_handlers.h>
#include <yxiva/core/resharding/migration.h>
#include <yxiva/core/gid.h>
#include <yxiva/core/operation_result.h>
#include <yxiva/core/types.h>
#include <ymod_pq/call.h>
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/spinlock.h>

namespace yxiva { namespace resharding {

template <typename PQcall>
class pq_migration_storage
    : public yplatform::module
    , public migration_storage
{
    using pq_call = PQcall;

public:
    struct settings
    {
        string conninfo;
        string get_ranges_request;
        string set_state_request;
        time_duration update_interval;

        void load(const yplatform::ptree& conf)
        {
            conninfo = conf.get<string>("conninfo");
            get_ranges_request = conf.get<string>("get_ranges_request", "get_migrations_range");
            set_state_request = conf.get<string>("set_state_request", "set_migration_state");
            update_interval = conf.get<time_duration>("update_interval");
        }
    };

    pq_migration_storage<pq_call>(const yplatform::ptree& conf)
    {
        impl_ = load_impl(conf);
    }

    void reload(const yplatform::ptree& conf)
    {
        auto new_impl = load_impl(conf);
        auto old_impl = get_impl();

        old_impl->stop();
        new_impl->start();

        set_impl(new_impl);
    }

    void start()
    {
        get_impl()->start();
    }

    void stop()
    {
        get_impl()->stop();
    }

    virtual std::shared_ptr<const migrations> get() const override
    {
        return get_impl()->get();
    }

    template <typename Handler>
    void fetch_migrations(Handler&& handler)
    {
        get_impl()->fetch_migrations(std::forward<Handler>(handler));
    }

    template <typename Handler>
    void set_migration_state(gid_t gid, migration::state_type state, Handler&& handler)
    {
        get_impl()->set_migration_state(gid, state, std::forward<Handler>(handler));
    }

private:
    class impl;

    std::shared_ptr<impl> load_impl(const yplatform::ptree& conf)
    {
        auto reactor = yplatform::find_reactor(conf.get("reactor", "global"));

        auto pq_name = conf.get<string>("pq", "migrations_pq");
        auto pq = yplatform::find<pq_call, std::shared_ptr>(pq_name);

        settings s;
        s.load(conf);
        return std::make_shared<impl>(*reactor->io(), s, pq);
    }

    std::shared_ptr<impl> get_impl() const
    {
        std::lock_guard<yplatform::spinlock> lock(spinlock_);
        return impl_;
    }

    void set_impl(std::shared_ptr<impl> new_impl)
    {
        std::lock_guard<yplatform::spinlock> lock(spinlock_);
        impl_ = std::move(new_impl);
    }

    mutable yplatform::spinlock spinlock_;
    std::shared_ptr<impl> impl_;

    friend class test_access;
};

template <typename PQcall>
class pq_migration_storage<PQcall>::impl
    : public std::enable_shared_from_this<pq_migration_storage<PQcall>::impl>
{
    using pq_result = yplatform::future::future<bool>;
    using self_type = pq_migration_storage<PQcall>::impl;

public:
    impl(boost::asio::io_service& io, const settings& settings, const std::shared_ptr<pq_call>& pq)
        : migrations_(std::make_shared<migrations>())
        , ctx_(boost::make_shared<yplatform::task_context>("pq_migration_storage_ctx"))
        , conninfo_(settings.conninfo)
        , get_ranges_request_(settings.get_ranges_request)
        , set_state_request_(settings.set_state_request)
        , update_interval_(settings.update_interval)
        , migrations_update_timer_(io)
        , strand_(io)
        , pq_(pq)
    {
    }

    void start()
    {
        running_ = true;
        get_migrations_async([this, self = yplatform::shared_from(this)](const operation::result&) {
            schedule_get_migrations_async();
        });
    }

    void stop()
    {
        strand_.post([this, self = yplatform::shared_from(this)]() {
            running_ = false;
            migrations_update_timer_.cancel();
        });
    }

    std::shared_ptr<const migrations> get() const
    {
        std::lock_guard<yplatform::spinlock> lock(spinlock_);
        return migrations_;
    }

    template <typename Handler>
    void fetch_migrations(Handler&& handler)
    {
        get_migrations_async(std::forward<Handler>(handler));
    }

    template <typename Handler>
    void set_migration_state(gid_t gid, migration::state_type state, Handler&& handler)
    {
        set_migration_state_async(gid, state, std::forward<Handler>(handler));
    }

private:
    template <typename Handler>
    void get_migrations_async(Handler&& handler)
    {
        if (!running_)
        {
            return;
        }

        auto args = boost::make_shared<ymod_pq::bind_array>();

        auto pq_handler = boost::make_shared<range_request_handler>();
        auto fres = pq_->request(ctx_, conninfo_, get_ranges_request_, args, pq_handler, true);
        fres.add_callback(strand_.wrap([this,
                                        self = yplatform::shared_from(this),
                                        fres,
                                        pq_handler,
                                        h = std::forward<Handler>(handler)]() {
            on_range_request_response(fres, pq_handler, h);
        }));
    }

    template <typename Handler>
    void set_migration_state_async(gid_t gid, migration::state_type state, Handler&& handler)
    {
        if (!running_)
        {
            return;
        }

        // This variable is necessary, because push_swap_int64_vector
        // accepts lvalue reference, not rvalue.
        auto allowed_states_arg = allowed_states(state);

        auto args = boost::make_shared<ymod_pq::bind_array>();
        ymod_pq::push_const_uint(args, gid);
        ymod_pq::push_swap_int64_vector(args, allowed_states_arg);
        ymod_pq::push_const_int(args, static_cast<int>(state));

        auto pq_handler = boost::make_shared<range_request_handler>();
        auto fres = pq_->request(ctx_, conninfo_, set_state_request_, args, pq_handler, true);
        fres.add_callback(strand_.wrap([this,
                                        self = yplatform::shared_from(this),
                                        gid,
                                        fres,
                                        pq_handler,
                                        h = std::forward<Handler>(handler)]() mutable {
            on_set_request_response(gid, fres, pq_handler, h);
        }));
    }

    void schedule_get_migrations_async()
    {
        if (!running_)
        {
            return;
        }

        migrations_update_timer_.expires_from_now(update_interval_);
        migrations_update_timer_.async_wait(strand_.wrap(
            [this, self = yplatform::shared_from(this)](const boost::system::error_code& ec) {
                if (ec == boost::asio::error::operation_aborted)
                {
                    return;
                }

                get_migrations_async(
                    [this, self](const operation::result&) { schedule_get_migrations_async(); });
            }));
    }

    template <typename Handler>
    void on_range_request_response(
        const pq_result& future,
        const boost::shared_ptr<range_request_handler>& pq_handler,
        Handler&& h)
    {
        auto reply_status = check_pq_response(future, pq_handler);
        if (reply_status)
        {
            set_migrations(pq_handler->move_data());
        }
        else
        {
            YLOG_CTX_GLOBAL(ctx_, error)
                << "migrations storage fail: " << reply_status.error_reason;
        }
        h(reply_status);
    }

    operation::result get_state(gid_t gid, const migrations& m, migration::state_type& state)
    {
        auto it = std::lower_bound(
            m.begin(), m.end(), migration{ migration::state_type::pending, 0, gid });
        if (it == m.end())
        {
            return "could not find migration for gid " + std::to_string(gid);
        }
        state = it->state;
        return operation::success;
    }

    template <typename Handler>
    void on_set_request_response(
        gid_t gid,
        const pq_result& future,
        const boost::shared_ptr<range_request_handler>& pq_handler,
        Handler&& h)
    {
        migration::state_type state_after;
        auto reply_status = check_pq_response(future, pq_handler, gid, state_after);
        if (reply_status)
        {
            set_migrations(pq_handler->move_data());
        }
        else
        {
            YLOG_CTX_GLOBAL(ctx_, error)
                << "migrations storage fail: " << reply_status.error_reason;
        }
        h(reply_status, state_after, pq_handler->bad_state());
    }

    operation::result check_pq_response(
        const pq_result& future,
        const boost::shared_ptr<range_request_handler>& pq_handler)
    {
        try
        {
            future.get();
        }
        catch (const std::exception& e)
        {
            return e.what();
        }
        return pq_handler->failure_reason();
    }

    operation::result check_pq_response(
        const pq_result& future,
        const boost::shared_ptr<range_request_handler>& pq_handler,
        gid_t gid,
        migration::state_type& state_after)
    {
        try
        {
            future.get();
        }
        catch (const std::exception& e)
        {
            return e.what();
        }
        if (pq_handler->failure_reason().size())
        {
            return pq_handler->failure_reason();
        }
        return get_state(gid, pq_handler->data(), state_after);
    }

    void set_migrations(migrations&& v)
    {
        auto new_migrations = std::make_shared<migrations>(std::move(v));
        std::lock_guard<yplatform::spinlock> lock(spinlock_);
        std::swap(migrations_, new_migrations);
    }

    std::vector<int64_t> allowed_states(migration::state_type next_state)
    {
        using s = migration::state_type;

        switch (next_state)
        {
        case s::pending:
            return { static_cast<int64_t>(s::ready), static_cast<int64_t>(s::inprogress) };
        case s::ready:
            return { static_cast<int64_t>(s::pending) };
        case s::inprogress:
            return { static_cast<int64_t>(s::ready) };
        case s::finished:
            return { static_cast<int64_t>(s::inprogress) };
        }

        return {};
    }

    mutable yplatform::spinlock spinlock_;
    std::shared_ptr<migrations> migrations_;

    task_context_ptr ctx_;
    string conninfo_;
    string get_ranges_request_;
    string set_state_request_;
    time_duration update_interval_;

    time_traits::timer migrations_update_timer_;
    boost::asio::io_service::strand strand_;

    bool running_ = false;

    std::shared_ptr<pq_call> pq_;
};

}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::resharding::pq_migration_storage<ymod_pq::call>)
