#pragma once

#include <yxiva/core/resharding/migration_storage.h>
#include <yxiva/core/shards/storage.h>
#include <yplatform/spinlock.h>
#include <yplatform/module.h>
#include <yplatform/find.h>

namespace yxiva { namespace shard_config {

class merging_storage
    : public yplatform::module
    , public storage
{
public:
    merging_storage(const yplatform::ptree& config)
    {
        load_storages(config);
    }

    void reload(const yplatform::ptree& config)
    {
        load_storages(config);
    }

    virtual std::shared_ptr<const shards> get() const
    {
        return get_config();
    }

    std::shared_ptr<storage> shards_from() const
    {
        return get_settings()->shards_from;
    }

    std::shared_ptr<storage> shards_to() const
    {
        return get_settings()->shards_to;
    }

private:
    struct gid_range
    {
        gid_t start_gid;
        gid_t end_gid;

        bool operator<(const gid_range& rhs) const
        {
            return end_gid < rhs.end_gid;
        }
    };

    class empty_storage : public storage
    {
    public:
        virtual std::shared_ptr<const shards> get() const override
        {
            // This contraption maps every gid to nowhere;
            // emulates unaccessible shard storage,
            // used during resharding when migration storage is unavailable.
            return std::make_shared<const shards>();
        }
    };

    using storages = std::map<gid_range, std::shared_ptr<storage>>;

    struct settings
    {
        storages static_storages;
        bool enable_resharding;
        std::shared_ptr<resharding::migration_storage> migrations;
        std::shared_ptr<storage> shards_from;
        std::shared_ptr<storage> shards_to;
    };

    void load_storages(const yplatform::ptree& config)
    {
        settings new_settings;

        auto resharding_config = config.get_child_optional("resharding");
        if (resharding_config)
        {
            new_settings.enable_resharding = true;
            load_resharding_config(*resharding_config, new_settings);
        }
        else
        {
            new_settings.enable_resharding = false;
            load_ranges(config, new_settings);
        }

        set_settings(std::make_shared<settings>(std::move(new_settings)));
    }

    void load_resharding_config(const yplatform::ptree& config, settings& s)
    {
        s.migrations = yplatform::find<resharding::migration_storage, std::shared_ptr>(
            config.get<string>("migrations"));
        s.shards_from =
            yplatform::find<storage, std::shared_ptr>(config.get<string>("shards_from"));
        s.shards_to = yplatform::find<storage, std::shared_ptr>(config.get<string>("shards_to"));
    }

    void load_ranges(const yplatform::ptree& config, settings& s)
    {
        gid_t start_gid = 0;
        auto range = config.equal_range("ranges");
        for (auto it = range.first; it != range.second; ++it)
        {
            auto& storage_node = it->second;

            gid_t end_gid = storage_node.get<gid_t>("max_gid");
            gid_range new_range{ start_gid, end_gid };
            string name = storage_node.get<string>("shards");
            auto source_storage = yplatform::find<storage, std::shared_ptr>(name);
            s.static_storages[new_range] = source_storage;

            start_gid = end_gid + 1;
        }
    }

    std::shared_ptr<const shards> get_config() const
    {
        auto s = get_settings();

        shards merged_shards;
        storages dynamic_storages = get_dynamic_storages(*s);
        auto& current_storages = s->enable_resharding ? dynamic_storages : s->static_storages;

        id_t shard_id = 0;
        for (auto& range_storage_pair : current_storages)
        {
            auto& range = range_storage_pair.first;
            auto& storage = range_storage_pair.second;
            auto source_shards = storage->get();
            append_configs(range, shard_id, *source_shards, merged_shards);
        }
        return std::make_shared<const shards>(std::move(merged_shards));
    }

    storages get_dynamic_storages(const settings& s) const
    {
        if (!s.enable_resharding)
        {
            return {};
        }

        storages dynamic_storages;
        auto migrations = s.migrations->get();
        if (migrations->empty())
        {
            gid_range full_range{ 0, 65535 };
            dynamic_storages[full_range] = std::make_shared<empty_storage>();
            return dynamic_storages;
        }

        for (auto& m : *migrations)
        {
            gid_range range{ m.start_gid, m.end_gid };
            dynamic_storages[range] = storage_from_state(m.state, s);
        }
        return dynamic_storages;
    }

    std::shared_ptr<storage> storage_from_state(
        resharding::migration::state_type state,
        const settings& s) const
    {
        switch (state)
        {
        case resharding::migration::state_type::pending:
        case resharding::migration::state_type::ready:
        case resharding::migration::state_type::inprogress:
            return s.shards_from;

        case resharding::migration::state_type::finished:
            return s.shards_to;
        }
    }

    void append_configs(
        gid_range range,
        id_t& shard_id,
        const shards& source_shards,
        shards& dest_shards) const
    {
        for (auto& source_shard : source_shards)
        {
            if (source_shard.end_gid < range.start_gid || source_shard.start_gid > range.end_gid)
            {
                continue;
            }

            dest_shards.push_back(
                shard_config::shard{ shard_id,
                                     std::max(source_shard.start_gid, range.start_gid),
                                     std::min(source_shard.end_gid, range.end_gid),
                                     source_shard.master,
                                     source_shard.replicas });
            ++shard_id;
        }
    }

    void set_settings(std::shared_ptr<settings>&& s)
    {
        std::lock_guard<yplatform::spinlock> l(lock_);
        settings_ = std::move(s);
    }

    const std::shared_ptr<settings> get_settings() const
    {
        std::lock_guard<yplatform::spinlock> l(lock_);
        return settings_;
    }

    mutable yplatform::spinlock lock_;
    std::shared_ptr<settings> settings_;
};

}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::shard_config::merging_storage)
