#pragma once

#include <yxiva/core/types.h>
#include <yxiva/core/shards/shards.h>

namespace yxiva { namespace shard_config {

template <typename PQResponseHandler>
class pq_request_handler : public PQResponseHandler
{
    enum class db_role
    {
        master,
        replica
    };

    const string conninfo_params_;

    unsigned current_id_;
    db_role current_role_;
    string current_conninfo_;

    string failure_reason_;

    db_role db_role_from_string(const string& s)
    {
        if (s == "master") return db_role::master;
        if (s == "replica") return db_role::replica;

        throw std::invalid_argument("invalid db_role string: " + s);
    }

public:
    enum column
    {
        id,
        role,
        conninfo,

        cnt
    };

    struct shard_instances
    {
        db_instance master;
        db_instances replicas;
    };

    pq_request_handler(const string& conninfo_params) : conninfo_params_(conninfo_params)
    {
    }

    virtual void handle_cell(unsigned /*row*/, unsigned col, const string& value, bool is_null)
        override
    {
        if (!failure_reason_.empty())
        {
            return;
        }

        if (is_null)
        {
            failure_reason_ = "shard parameter " + std::to_string(col) + " is null";
            return;
        }

        try
        {
            switch (col)
            {
            case column::id:
                current_id_ = boost::lexical_cast<id_t>(value);
                break;
            case column::role:
                current_role_ = db_role_from_string(value);
                break;
            case column::conninfo:
                current_conninfo_ = value + " " + conninfo_params_;
                break;
            default:
                throw std::runtime_error("unknown pq column in pq_updater!");
            }
        }
        catch (const std::exception& e)
        {
            failure_reason_ =
                string("invalid field values (or number of fields) in pq response: ") + e.what();
        }
    }

    void handle_row_end(unsigned /*row*/) override
    {
        if (!failure_reason_.empty())
        {
            return;
        }

        switch (current_role_)
        {
        case db_role::master:
            result_[current_id_].master.conninfo = current_conninfo_;
            break;
        case db_role::replica:
            result_[current_id_].replicas.emplace_back(
                shard_config::db_instance{ current_conninfo_ });
            break;
        }

        current_id_ = unsigned{};
        current_role_ = db_role{};
        current_conninfo_ = string{};
    }

    unsigned column_count() const override
    {
        return column::cnt;
    }

    const string& failure_reason() const
    {
        return failure_reason_;
    }

    std::unordered_map<unsigned, shard_instances> move_data()
    {
        return std::move(result_);
    }

private:
    std::unordered_map<unsigned, shard_instances> result_;
};

}}
