#pragma once

#include <yxiva/core/shards/storage.h>
#include <yxiva/core/shards/pq_handler.h>
#include <yplatform/reactor.h>
#include <yplatform/find.h>
#include <yplatform/spinlock.h>
#include <yplatform/module.h>

namespace yxiva { namespace shard_config {

template <typename PQCall, typename PQResult, typename PQResponseHandler>
class pq_updated_storage
    : public yplatform::module
    , public storage
{
public:
    using pq_call = PQCall;
    using pq_result = PQResult;
    using pq_response_handler = PQResponseHandler;
    using this_type = pq_updated_storage<pq_call, pq_result, pq_response_handler>;

    struct settings
    {
        time_duration update_interval;
        string query;
        string conninfo_params;
        string plproxy_conninfo;

        void load(const yplatform::ptree& conf)
        {
            update_interval = conf.get<time_duration>("update_interval");
            query = conf.get("query", "list_shards");
            conninfo_params = conf.get("conninfo_params", "");
            plproxy_conninfo = conf.get<string>("plproxy_conninfo");
        }
    };

    virtual void init(const yplatform::ptree& conf)
    {
        impl_ = load_impl(conf);
    }

    virtual void reload(const yplatform::ptree& conf)
    {
        auto new_impl = load_impl(conf);
        auto old_impl = get_impl();

        old_impl->stop();
        new_impl->start();

        set_impl(std::move(new_impl));
    }

    void start()
    {
        get_impl()->start();
    }

    void stop()
    {
        get_impl()->stop();
    }

    virtual std::shared_ptr<const shards> get() const override
    {
        return get_impl()->get();
    }

protected:
    class impl;

private:
    using gid_mapping = std::map<gid_t, unsigned>; // end_gid <-> id

    std::shared_ptr<impl> load_impl(const yplatform::ptree& conf)
    {
        gid_mapping gid_mapping = load_mapping(conf);
        auto reactor = yplatform::global_reactor_set->get(conf.get("reactor", "global"));
        string pq_name = conf.get("pq", "pq_shards");
        auto pq = yplatform::find<pq_call, std::shared_ptr>(pq_name);
        settings settings;
        settings.load(conf);

        return std::make_shared<impl>(*reactor->io(), settings, gid_mapping, pq);
    }

    gid_mapping load_mapping(const yplatform::ptree& conf)
    {
        gid_mapping gid_mapping;
        auto range = conf.equal_range("mapping");
        for (auto it = range.first; it != range.second; ++it)
        {
            auto id = it->second.get<unsigned>("id");
            auto gid = it->second.get<gid_t>("max_gid");
            gid_mapping[gid] = id;
        }
        if (gid_mapping.empty())
        {
            throw std::runtime_error("no ranges specified for shard gid<->id mapping");
        }
        return gid_mapping;
    }

    std::shared_ptr<impl> get_impl() const
    {
        std::lock_guard<yplatform::spinlock> lock(spinlock_);
        return impl_;
    }

    void set_impl(std::shared_ptr<impl>&& new_impl)
    {
        std::lock_guard<yplatform::spinlock> lock(spinlock_);
        impl_ = std::move(new_impl);
    }

    mutable yplatform::spinlock spinlock_;
    std::shared_ptr<impl> impl_;
};

template <typename PQCall, typename PQResult, typename PQResponseHandler>
class pq_updated_storage<PQCall, PQResult, PQResponseHandler>::impl
    : public std::enable_shared_from_this<impl>
{
public:
    impl(
        boost::asio::io_service& io,
        const settings& settings,
        const gid_mapping& gid_mapping,
        const std::shared_ptr<pq_call>& pq)
        : settings_(settings)
        , gid_mapping_(gid_mapping)
        , shards_(std::make_shared<shards>())
        , ctx_(boost::make_shared<yplatform::task_context>("pq shard update ctx"))
        , pq_(pq)
        , strand_(io)
        , delay_timer_(io)
    {
    }

    void start()
    {
        strand_.post([this, shared_this = this->shared_from_this()]() {
            running_ = true;
            call_protected([this, shared_this]() { request_shards(); });
        });
    }

    void stop()
    {
        strand_.post([this, shared_this = this->shared_from_this()]() {
            running_ = false;
            boost::system::error_code ec;
            delay_timer_.cancel(ec);
            if (ec)
            {
                YLOG_CTX_GLOBAL(ctx_, error) << "timer cancel error: " << ec.message();
            }
        });
    }

    std::shared_ptr<const shards> get() const
    {
        return get_shards();
    }

private:
    using pq_handler_t = pq_request_handler<pq_response_handler>;
    using instances_by_id = std::unordered_map<unsigned, typename pq_handler_t::shard_instances>;

    void request_shards()
    {
        auto pq_handler = boost::make_shared<pq_handler_t>(settings_.conninfo_params);
        pq_result fres = pq_->request(
            ctx_, settings_.plproxy_conninfo, settings_.query, nullptr, pq_handler, true);
        fres.add_callback(wrap([this, shared_this = this->shared_from_this(), fres, pq_handler]() {
            on_pq_response(fres, pq_handler);
        }));
    }

    void on_pq_response(const pq_result& future, const boost::shared_ptr<pq_handler_t>& pq_handler)
    {
        bool pq_result_valid = true;
        try
        {
            future.get();
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_GLOBAL(ctx_, error) << "shard storage fail: " << e.what();
            pq_result_valid = false;
        }

        if (!pq_handler->failure_reason().empty())
        {
            YLOG_CTX_GLOBAL(ctx_, error)
                << "invalid reply from pq: " << pq_handler->failure_reason();
            pq_result_valid = false;
        }

        if (pq_result_valid)
        {
            set_shards(
                std::make_shared<const shards>(make_shards(gid_mapping_, pq_handler->move_data())));
        }

        request_after_delay();
    }

    void request_after_delay()
    {
        auto handler =
            [this, shared_this = this->shared_from_this()](const boost::system::error_code& ec) {
                if (ec == boost::asio::error::operation_aborted)
                {
                    return;
                }
                request_shards();
            };

        delay_timer_.expires_from_now(settings_.update_interval);
        delay_timer_.async_wait(
            wrap<decltype(handler), const boost::system::error_code&>(std::move(handler)));
    }

    template <typename Function, typename... Args>
    std::function<void(Args...)> wrap(Function&& f)
    {
        return strand_.wrap(
            [this, shared_this = this->shared_from_this(), f_moved = std::move(f)](Args... args) {
                call_protected(std::move(f_moved), std::forward<Args>(args)...);
            });
    }

    template <typename Function, typename... Args>
    void call_protected(Function&& f, Args&&... args)
    {
        if (!running_)
        {
            return;
        }

        try
        {
            f(std::forward<Args>(args)...);
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_GLOBAL(ctx_, error) << "top-level exception: " << e.what() << "restarting in "
                                         << time_traits::to_string(settings_.update_interval);
            try
            {
                request_after_delay();
            }
            catch (...)
            {
                abort();
            }
        }
    }

    shards make_shards(const gid_mapping& gid_mapping, instances_by_id instances)
    {
        shards shards;
        gid_t start_gid = 0;
        for (auto& gid_id_pair : gid_mapping)
        {
            auto end_gid = gid_id_pair.first;
            auto id = gid_id_pair.second;

            shard shard;
            shard.id = id;
            shard.start_gid = start_gid;
            shard.end_gid = end_gid;
            start_gid = end_gid + 1;

            auto it = instances.find(id);
            if (it == instances.end())
            {
                YLOG_CTX_GLOBAL(ctx_, error) << "no instances for shard " << shard.describe();
                continue;
            }

            shard.master = std::move(it->second.master);
            shard.replicas = std::move(it->second.replicas);

            shards.push_back(std::move(shard));
        }
        return shards;
    }

    void set_shards(std::shared_ptr<const shards>&& shards)
    {
        std::lock_guard<yplatform::spinlock> l(lock_);
        shards_ = std::move(shards);
    }

    std::shared_ptr<const shards> get_shards() const
    {
        std::lock_guard<yplatform::spinlock> l(lock_);
        return shards_;
    }

    const settings settings_;
    const gid_mapping gid_mapping_;

    std::shared_ptr<const shards> shards_;
    mutable yplatform::spinlock lock_;

    task_context_ptr ctx_;
    bool running_{ false };

    std::shared_ptr<pq_call> pq_;

    boost::asio::io_service::strand strand_;
    time_traits::timer delay_timer_;
};

}}
