#pragma once

#include <yxiva/core/shards/storage.h>
#include <yplatform/spinlock.h>
#include <yplatform/module.h>

namespace yxiva { namespace shard_config {

class static_storage
    : public yplatform::module
    , public storage
{
public:
    virtual void init(const yplatform::ptree& config)
    {
        load_shards(config);
    }

    virtual void reload(const yplatform::ptree& config)
    {
        load_shards(config);
    }

    virtual std::shared_ptr<const shards> get() const override
    {
        return get_shards();
    }

private:
    void load_shards(const yplatform::ptree& config)
    {
        auto config_shards = std::make_shared<shards>();
        gid_t start_gid = 0;

        auto range = config.equal_range("shards");
        for (auto it = range.first; it != range.second; ++it)
        {
            auto& shard_node = it->second;
            shard shard;
            shard.id = shard_node.get<id_t>("id");

            shard.start_gid = start_gid;
            shard.end_gid = shard_node.get<gid_t>("max_gid");
            start_gid = shard.end_gid + 1;

            if (auto multirole = shard_node.get_child_optional("multirole_conninfo"))
            {
                load_multirole_conninfo(multirole.get(), shard);
            }
            else
            {
                load_master_replicas(shard_node, shard);
            }

            config_shards->push_back(std::move(shard));
        }

        set_shards(std::move(config_shards));
    }

    void load_multirole_conninfo(const yplatform::ptree& node, shard& shard)
    {
        string hosts;
        auto hosts_range = node.equal_range("hosts");
        for (auto it = hosts_range.first; it != hosts_range.second; ++it)
        {
            hosts += it->second.data() + ',';
        }
        if (hosts.empty())
        {
            throw std::invalid_argument("must have at least one host in multirole conninfo");
        }
        // Cut off the trailing ','.
        hosts.resize(hosts.size() - 1);
        hosts = "host=" + hosts;
        string params = node.get<string>("params");
        string master_params = params + " " + node.get<string>("master_params");
        string replica_params = params + " " + node.get<string>("replica_params", "");
        shard.master = { hosts + " " + master_params };
        shard.replicas.push_back({ hosts + " " + replica_params });
    }

    void load_master_replicas(const yplatform::ptree& node, shard& shard)
    {
        shard.master = { node.get<string>("master") };
        auto replicas_range = node.equal_range("replicas");
        for (auto it = replicas_range.first; it != replicas_range.second; ++it)
        {
            shard.replicas.push_back({ it->second.data() });
        }
    }

    void set_shards(std::shared_ptr<shards>&& shards)
    {
        std::lock_guard<yplatform::spinlock> l(lock_);
        shards_ = std::move(shards);
    }

    const std::shared_ptr<shards> get_shards() const
    {
        std::lock_guard<yplatform::spinlock> l(lock_);
        return shards_;
    }

    mutable yplatform::spinlock lock_;
    std::shared_ptr<shards> shards_;
};

}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(yxiva::shard_config::static_storage)
