#pragma once

#include <yxiva/core/types.h>
#include <boost/algorithm/string.hpp>
#include <algorithm>

namespace yxiva {

struct root_params_t
{
    string uid;
    string service;
    MSGPACK_DEFINE(uid, service);
};

struct sub_params_t
{
    string id;
    string filter;
    string callback_url;
    string extra_data;
    string client;
    string session_key;
    ttl_t ttl = 0;

    local_id_t init_local_id = 0;
    time_t init_time = 0;
    local_id_t ack_local_id = 0;
    time_t ack_time = 0;
    bool smart_notify = false;
    string platform; // mobile platform name
    string device;
    string bb_connection_id;
    string uidset;
    time_t next_retry_time = 0;
    time_traits::duration::rep retry_interval = 0;
    time_t ack_event_ts = 0;

    sub_params_t() = default;

    sub_params_t(
        string _id,
        string _filter,
        string _callback_url,
        string _extra_data,
        string _client,
        string _session_key,
        ttl_t _ttl)
        : id(_id)
        , filter(_filter)
        , callback_url(_callback_url)
        , extra_data(_extra_data)
        , client(_client)
        , session_key(_session_key)
        , ttl(_ttl)
    {
    }

    sub_params_t(
        string _id,
        string _filter,
        string _callback_url,
        string _extra_data,
        string _client,
        string _session_key,
        ttl_t _ttl,
        local_id_t _init_local_id,
        time_t _init_time)
        : id(_id)
        , filter(_filter)
        , callback_url(_callback_url)
        , extra_data(_extra_data)
        , client(_client)
        , session_key(_session_key)
        , ttl(_ttl)
        , init_local_id(_init_local_id)
        , init_time(_init_time)
        , ack_local_id(_init_local_id)
        , ack_time(_init_time)
    {
    }

    string to_string() const
    {
        return id + " " + client + " " + session_key + " " + callback_url;
    }

    MSGPACK_DEFINE(
        id,
        filter,
        callback_url,
        extra_data,
        client,
        session_key,
        ttl,
        init_local_id,
        init_time,
        ack_local_id,
        ack_time,
        smart_notify,
        platform,
        device,
        bb_connection_id,
        uidset,
        next_retry_time,
        retry_interval,
        ack_event_ts);
};

struct sub_t
    : root_params_t
    , sub_params_t
{
    sub_t()
    {
    }
    sub_t(const sub_params_t& params) : root_params_t(), sub_params_t(params)
    {
    }
    // could be unpacked to subscription_t
    MSGPACK_DEFINE(MSGPACK_BASE(root_params_t), MSGPACK_BASE(sub_params_t));
};

inline std::ostream& operator<<(std::ostream& stream, const sub_t& subscription)
{
    stream << "(" << subscription.uid << ", " << subscription.service << ", [";
    stream << subscription.filter;
    stream << "], " << subscription.id << ", " << subscription.callback_url << ", "
           << subscription.extra_data << ", " << subscription.client << ", " << subscription.ttl
           << ")";

    return stream;
}

inline bool operator<(const root_params_t& a, const root_params_t& b)
{
    return a.uid < b.uid || (a.uid == b.uid && a.service < b.service);
}

typedef std::vector<sub_t> sub_list;

// Remove delimiters and switch characters to lower case.
inline string canonize_device_id(string source)
{
    source.erase(std::remove(source.begin(), source.end(), '-'), source.end());
    boost::algorithm::to_lower(source);
    return source;
}

}
