#!/bin/bash

# autocleanup on ctrl+c and script errors
trap "{ cleanup; exit 1; }" INT ERR
trap "{ cleanup; }" INT EXIT

function pushd {
    command pushd "$@" > /dev/null
}

function popd {
    command popd "$@" > /dev/null
}

function is_alive {
    pids=$@
    for pid in ${pids[@]}; do
        if ! kill -0 $pid &> /dev/null; then
            return 1
        fi
    done
    return 0
}

function run_app {
    pushd $1
    $2 ${@:3} &>/dev/null &
    pid=$!
    popd

    sleep 1
    if ! is_alive $pid; then
        return 1
    fi

    echo $pid
    return 0
}

function stop_app {
    pids=$@
    for pid in ${pids[@]}; do
        if [[ "$pid" != "" ]]; then
            kill -TERM $pid &>/dev/null || continue;
            for i in {0..30}; do
                if ! is_alive $pid; then
                    break;
                fi
                sleep 0.1
            done
            if is_alive $pid; then
                kill -9 $pid &>/dev/null
            fi
        fi
    done
    return 0
}

function cleanup {
    stop_app $pids;
}

function wait_pg_is_ready {
    host=$1
    port=$2
    dbname=$3
    user=$4
    password=$5
    timeout=30
    echo -n "waiting pg $host:$port."
    for i in $(seq 1 $timeout); do
        PGPASSWORD="$password" psql --host=$host --port=$port --dbname=$dbname --username=$user -c '\q' &> /dev/null  && break
        [[ "$i" == "$timeout" ]] && (echo " timeout"; exit 1)
        echo -n "."; sleep 1
    done
    echo ' ready'
    return 0
}

function wait_for_ping {
    echo -ne 'wait for ping'
    endpoint=$1
    for i in {0..20};
    do
        if (curl "${endpoint}/ping" | grep pong -q) &>/dev/null; then
            echo
            return 0
        fi
        echo -ne '.'
        sleep 0.2
    done
    echo
    echo "ping not successful"
    exit 1
}