import urllib
import httplib
import json
import msgpack
import io

from wait_condition import wait

class RawClient:
    def __init__(self, host, port):
        self.__host = host
        self.__port = port
        self.__connection = httplib.HTTPConnection(host, port)

    def get(self, url, parameters = [], headers = {}):
        return self._request("GET", url, parameters, "", headers)

    def post(self, url, parameters = [], body = "", headers = {}):
        return self._request("POST", url, parameters, body, headers)

    def _request(self, method, url, parameters, body, headers):
        uri=url
        if len(parameters)!=0:
            uri=uri+"?"+urllib.urlencode(parameters)
        self.__connection.request(method, uri, body, headers)
        print('request: %s %s %s %s' % (method, url, '\n'.join(['%s: %s' % (key, headers[key]) for key in headers]), body))
        resp = self.__connection.getresponse()
        resp.body = resp.read(resp.getheader("content-length"))
        resp.headers = self.array_to_dict(resp.getheaders())
        print('reponse: %d %s, ctx: %s' % (resp.status, resp.body, resp.headers["y-context"]))
        return resp

    def array_to_dict(self, array):
        result = dict()
        for a in array:
            result[a[0]] = a[1]
        return result

class Client:
    def __init__(self, host, port):
        self.__host = host
        self.__port = port
        self.__raw = RawClient(host, port)

    def get_200(self, url, parameters = {}, headers = {}):
        response = self.__raw.get(url, parameters, headers)
        if response.status != 200:
            raise ValueError(url + ' failed: ' + str(response.status) + ' ' + str(response.body))
        return response

    def post_200(self, url, parameters = [], body = "", headers = {}):
        response = self.__raw.post(url, parameters, body, headers)
        if response.status != 200:
            raise ValueError(url + ' failed: ' + str(response.status) + ' ' + str(response.body))
        return response


    def list(self, uid, service, **kwds):
        return self.list_impl(uid, service, 'master', **kwds)

    def list_replica(self, uid, service, **kwds):
        return self.list_impl(uid, service, 'replica', **kwds)

    def list_impl(self, uid, service, db_role, **kwds):
        params = {'uid':uid, 'service':service, 'db_role': db_role}
        params.update(kwds)
        response = self.__raw.get('/list_json', params)
        if response.status != 200:
            raise ValueError('list failed: ' + str(response.status) + ' ' + str(response.body))
        return json.loads(response.body)

    def list_uidset(self, uidset, service, **kwds):
        params = {'uidset':uidset, 'service':service}
        params.update(kwds)
        response = self.__raw.get('/uidset/list', params)
        if response.status != 200:
            raise ValueError('list_uidset failed: ' + str(response.status) + ' ' + str(response.body))
        return json.loads(response.body)

    def list_ids(self, uid, service, **kwds):
        list_subs = self.list(uid, service, **kwds)
        return map(lambda x: x['id'], list_subs)

    def replica_has_subscription(self, uid, service, id):
        subs = self.list_replica(uid, service)
        return any([s for s in subs if s['id'] == id])

    def replica_has_uidset_subscription(self, uidset, service, id):
        subs = self.list_uidset(uidset, service)
        return any([s for s in subs if s['id'] == id])

    def subscribe_ex(self, uid, service, callback, **kwds):
        params = {'uid':uid, 'service':service, 'callback':callback}
        params.update(kwds)
        response = self.get_200("/subscribe", params)
        wait(lambda: self.replica_has_subscription(uid, service, response.body), 0.5)
        return response

    def subscribe(self, uid, service, callback, **kwds):
        return self.subscribe_ex(uid, service, callback, **kwds).body

    def subscribe_mobile(self, uid, service, callback, **kwds):
        params = {'uid':uid, 'service':service, 'callback':callback}
        params.update(kwds)
        response = self.get_200("/subscribe_mobile", params)
        wait(lambda: self.replica_has_subscription(uid, service, response.body), 1.5)
        return response.body

    def subscribe_uidset(self, uidset, uid, service, callback, **kwds):
        params = {'uidset': uidset, 'uid':uid, 'service':service, 'callback':callback}
        params.update(kwds)
        response = self.get_200("/uidset/subscribe", params)
        wait(lambda: self.replica_has_uidset_subscription(uidset, service, response.body), 1.5)
        return response.body

    def unsubscribe(self, uid, service, subid):
        params = {'uid':uid, 'service':service, 'subscription-id':subid}
        response = self.get_200("/unsubscribe", params)
        wait(lambda: not self.replica_has_subscription(uid, service, subid), 1.5)

    def unsubscribe_uidset(self, uidset, uid, service, subid):
        params = {'uidset': uidset, 'uid':uid, 'service':service, 'subscription-id':subid}
        response = self.get_200("/uidset/unsubscribe", params)
        wait(lambda: not self.replica_has_subscription(uid, service, subid), 1.5)

    def notify(self, uid, service, body, **kwds):
        params = {'uid' : uid, 'service' : service}
        params.update(kwds)
        response = self.post_200("/notify/" + service, params, body)

    def fast_notify(self, uid, service, body, **kwds):
        params = {'uid' : uid, 'service' : service}
        params.update(kwds)
        response = self.post_200("/fast_notify/" + service, params, body)

    def binary_notify(self, msg, **kwds):
        response = self.post_200("/binary_notify", kwds, msgpack.packb(msg))

    def fast_binary_notify(self, msg, **kwds):
        response = self.post_200("/fast_binary_notify", kwds, msgpack.packb(msg))

    def batch_binary_notify(self, message, keys, **kwds):
        body = msgpack.packb(message)
        body = body + msgpack.packb(keys)
        params = {}
        params.update(kwds)
        return msgpack.unpackb(self.post_200("/batch_binary_notify", params, body).body)

    def unsubscribe_all(self, uid, service):
        list = self.list(uid, service)
        list_ids = map(lambda x: x['id'], list)
        for subid in list_ids:
            self.unsubscribe(uid, service, subid)

    def make_message(self, uid, service, data, raw_data, transit_id, local_id, event_ts):
        return [uid, '', service, '', '', '', data, raw_data, True, transit_id, local_id, event_ts, [], {}]

    def make_subscription(self, uid, service, subid, callback):
        return [[uid, service], [subid, '', callback, '', '', '', 0, 0, 0, 0, 0, False, '']]

    def xtasks_create(self, uid, service, local_id):
        params = {'uid':uid, 'service':service, 'local_id':local_id}
        response = self.get_200("/xtasks_create", params)

    def xtasks_start(self, worker, count):
        response = self.get_200("/xtasks_start", {'worker':worker, 'count':count})
        return json.loads(response.body)

    def xtasks_fin(self, worker, task_id):
        response = self.get_200("/xtasks_fin", {'worker':worker, 'id':task_id})

    def xtasks_delay(self, worker, task_id, delay_sec):
        response = self.get_200("/xtasks_delay", {'worker':worker, 'id':task_id, 'delay_sec':delay_sec})

    def xtasks_cleanup_active(self, deadline_sec):
        response = self.get_200("/xtasks_cleanup_active", {'deadline_sec':deadline_sec})

    def xtasks_cleanup_workers(self, deadline_sec):
        response = self.get_200("/xtasks_cleanup_workers", {'deadline_sec':deadline_sec})

    def xtasks_wakeup_delayed(self):
        response = self.get_200("/xtasks_wakeup_delayed", {})

    def xtasks_counters(self):
        response = self.get_200("/xtasks_counters")
        return json.loads(response.body)

    def xtasks_summary(self):
        response = self.get_200("/xtasks_summary", {})
        return json.loads(response.body)

    def xtasks_clear(self):
        response = self.get_200("/xtasks_clear", {})

    def deduplicate(self, uid, service):
        response = self.get_200("/deduplicate_mobile_subscriptions",
            {'uid': uid, 'service': service})
        return response

    def enable(self, what):
        return self.get_200("/enable/" + what)

    def disable(self, what):
        return self.get_200("/disable/" + what)
    def prepare_migration(self, gid, role):
        return self.post_200("/resharding/xtable/prepare_migration", {'gid': gid, 'role': role})

    def start_migration(self, gid, role, reaction = 'delay'):
        return self.post_200("/resharding/xtable/start_migration", {'gid': gid, 'role': role, 'request_reaction': reaction})

    def finalize_migration(self, gid, role):
        return self.post_200("/resharding/xtable/finalize_migration", {'gid': gid, 'role': role})

    def abort_migration(self, gid, role):
        return self.post_200("/resharding/xtable/abort_migration", {'gid': gid, 'role': role})

    def replica_status(self):
        response = self.get_200("/replica_status")
        return json.loads(response.body)

    def is_control_leader(self):
        return self.replica_status()['control_leader'] == True

    def resharding_status(self):
        response = self.get_200("/resharding/xtable/status")
        return json.loads(response.body)

    def raw(self):
        return self.__raw

    def host(self):
        return self.__host

    def port(self):
        return self.__port

import os

class Testing:
    @staticmethod
    def host():
        return os.environ["XTEST_HOST"]
    @staticmethod
    def port():
        return int(os.environ["XTEST_PORT"])
    @staticmethod
    def port2():
        return int(os.environ["XTEST_PORT2"])
    @staticmethod
    def uid1():
        return "4000222428"
    @staticmethod
    def uid2():
        return "4002232303"
    @staticmethod
    def uid3():
        return "3423422241"
    @staticmethod
    def text_uid():
        return "textuid"
    @staticmethod
    def service1():
        return "fake"
    @staticmethod
    def service2():
        return "ppanel"
    @staticmethod
    def subscriber_port():
        return 9998
    @staticmethod
    def dead_subscriber_port():
        return 10000
    @staticmethod
    def mobile_port():
        return 9995
    @staticmethod
    def uidset1():
        return "4000222428"
    @staticmethod
    def uidset2():
        return "4002232303"
    @staticmethod
    def uidset1_uid1():
        return Testing.uidset1() + "+1"
    @staticmethod
    def uidset1_uid2():
        return Testing.uidset1() + "+2"
    @staticmethod
    def uidset2():
        return "4002232303"
    @staticmethod
    def uidset2_uid1():
        return Testing.uidset2() + "+1"
    @staticmethod
    def uidset2_uid2():
        return Testing.uidset2() + "+2"

def enum(**enums):
    return type('Enum', (), enums)

# TODO replace Testing.* by TestData.*
TestData = enum(
    json_payload = '{"key" : "value", "arr": [1,2,3]}',
    text_payload = '__text_payload__',
    binary_payload = '@@@#$^#%^#$@@',

    transit_id = 'transitid1',
    transit_id2 = 'transitid2',

    event_name = 'test-event'
)

MessageContentType = enum(unknown=0, json=1, binary=2)

MessageFields = enum(
    uid = 0,
    service = 2,
    operation = 3,
    lcn = 4,
    session_key = 5,
    data = 6,
    raw_data = 7,
    bright = 8,
    transit_id = 9,
    local_id = 10,
    event_ts = 11,
    tags = 12,
    repacking_rules = 13,
    ttl = 14,
    flags = 15,
    topic = 16,
    event_ts_ms = 17,
    subscription_matchers = 18,
    experiments = 19,
    type = 20,
    batch_size = 21,
    broadcast = 22,

    COUNT = 23
)

def make_message(uid,
        service,
        data={},
        raw_data = TestData.text_payload,
        transit_id = TestData.transit_id,
        local_id = 0,
        event_ts = 0,
        operation=TestData.event_name, tags=[],
        ttl=7*24*3600):
    ret = [
        uid,
        '',
        service,
        operation,
        '',
        '',
        data,
        raw_data,
        True,
        transit_id,
        local_id,
        event_ts,
        tags,
        {},
        ttl,
        0,
        '',
        0,
        [],
        [],
        MessageContentType.unknown,
        0,
        False
    ]
    if raw_data == TestData.binary_payload:
        ret[MessageFields.type] = MessageContentType.binary
    elif raw_data == TestData.json_payload:
        ret[MessageFields.type] = MessageContentType.json
    else:
        ret[MessageFields.type] = MessageContentType.unknown
    return ret

def unpack_binary_frame(data):
    unpacker = msgpack.Unpacker(io.BytesIO(data))
    msg_type = unpacker.unpack()
    msg_hdr = unpacker.unpack()
    msg_payload = data[1 + len(msgpack.packb(msg_hdr)):]
    return (msg_type, msg_hdr, msg_payload)

from nose.tools import assert_equals, assert_not_equal, assert_regexp_matches, assert_in, assert_not_in, assert_list_equal

def check(response, code):
    assert_equals(response.status, code)

def check(response, code, body):
    assert_equals(response.status, code)
    assert_equals(response.body, body)
