#!/usr/bin/env python2

import ssl
import httplib
import urllib
import json
import msgpack
from select import select
import struct
import types
import six
import time
import socket
import os
from asserts import *

class WebSocketMessage:
    def __init__(self, **kwds):
        self.__dict__.update(kwds)

class WebSocket:
    """ websocket session """
    def __init__(self, sock):
        self.sock = sock
        self.sock.setsockopt(socket.SOL_TCP, socket.TCP_NODELAY, 1)
        self.sock.setblocking(0)

    # select with a short t/o allows not to use nonblocking sockets
    def poll(self, delay=0.1):
        readable, writable, exceptional = select([self.sock], [], [], delay)
        return readable

    def cond_recv(self, n):
        # if not self.poll():
            # raise RuntimeError('no data to read')
        bytes = self.sock.recv(n)
        if len(bytes) < n:
            raise RuntimeError('insufficient bytes')
        return bytes

    def close(self):
        self.sock.close()

    def recv_message(self):
        if not self.poll(): return None

        header = self.sock.recv(2)
        if len(header) < 2:
            return None

        fin_bit = 0x8
        opcode_bits = 0xF0

        byte0 = ord(header[0])
        fin = byte0 & fin_bit
        opcode = (byte0 & opcode_bits >> 4)
        byte1 = ord(header[1])
        mask = byte1 & 0x80
        payload_len = byte1 & 0x7F

        if (payload_len == 126):
            extended_len = self.cond_recv(2)
            payload_len = struct.unpack("!H", extended_len)[0]
        elif (payload_len == 127):
            extended_len = self.cond_recv(8)
            if len(extended_len) < 8: return None
            payload_len = struct.unpack("!Q", extended_len)[0]

        status_code = 0
        payload = self.cond_recv(payload_len)
        if len(payload) < payload_len: return None
        if opcode == 8:
            status_code = struct.unpack("!H", payload[0:2])[0]
            payload = payload[2:]

        return WebSocketMessage(opcode=opcode, status_code=status_code, payload=payload)

    def send_message(self, payload):
        # payload = payload.encode('utf-8')
        mask = 0
        fin, rsv1, rsv2, rsv3, opcode = 1, 0, 0, 0, 0x01
        message = chr(fin << 7 | rsv1 << 6 | rsv2 << 5 | rsv3 << 4 | opcode)

        length = len(payload)
        if length < 0x7e:
            message += chr(length)
            message = six.b(message)
        elif length < 1 << 16:
            message += chr(0x7e)
            message = six.b(message)
            message += struct.pack("!H", length)
        else:
            message += chr(0x7f)
            message = six.b(message)
            message += struct.pack("!Q", length)
        message+=payload

        self.sock.send(message)

    def send_binary_message(self, payload):
        mask = 0
        fin, rsv1, rsv2, rsv3, opcode = 1, 0, 0, 0, 0x02
        message = chr(fin << 7 | rsv1 << 6 | rsv2 << 5 | rsv3 << 4 | opcode)

        length = len(payload)
        if length < 0x7e:
            message += chr(length)
            message = six.b(message)
        elif length < 1 << 16:
            message += chr(0x7e)
            message = six.b(message)
            message += struct.pack("!H", length)
        else:
            message += chr(0x7f)
            message = six.b(message)
            message += struct.pack("!Q", length)
        message+=payload

        self.sock.send(message)

def recv_message(self):
    size_str = self.read(2)
    while size_str[-2:] != b"\r\n":
        size_str += self.read(1)
    sz = int(size_str[:-2], 16)
    if sz == 0: return ''
    data = self.read(sz)
    self.read(2)
    return data[6:-4]

class HTTPRaw(object):
    """ GET/POST requests helper """
    def __init__(self, host, port, secure=False):
        self.host = host
        self.port = port
        self.secure = secure

    def create_https_connection(self):
        if sys.version_info >= (2, 7, 9, 'final', 0):
            return httplib.HTTPSConnection(self.host, self.port, context=ssl._create_unverified_context())
        else:
            # Legacy Python that doesn't verify HTTPS certificates by default
            return httplib.HTTPSConnection(self.host, self.port)

    def create_http_connection(self):
        return httplib.HTTPConnection(self.host, self.port)

    def create_connection(self):
        return self.create_https_connection() if self.secure else self.create_http_connection()

    def GET(self, url, headers):
        conn = self.create_connection()
        conn.request("GET", url, "", headers)
        print('request: GET %s %s' % (url, '\n'.join(['%s: %s' % (key, headers[key]) for key in headers])))
        resp = conn.getresponse()
        resp.body = resp.read(resp.getheader("content-length"))
        resp.code = resp.status
        resp.headers = self.array_to_dict(resp.getheaders())
        print('reponse: %d %s, ctx: %s' % (resp.status, resp.body, resp.headers["y-context"]))
        return resp

    def POST(self, url, headers={}, body=""):
        conn = self.create_connection()
        conn.request("POST", url, body, headers)
        print('request: POST %s %s' % (url, '\n'.join(['%s: %s' % (key, headers[key]) for key in headers])))
        resp = conn.getresponse()
        resp.body = resp.read(resp.getheader("content-length"))
        resp.code = resp.status
        resp.headers = self.array_to_dict(resp.getheaders())
        print('reponse: %d %s, ctx: %s' % (resp.status, resp.body, resp.headers["y-context"]))
        return resp

    def WS(self, url, headers = {}):
        headers.update({"Connection": "Upgrade", "Upgrade": "Websocket", "Sec-WebSocket-Key":"1NiuLqivRYatpZqwBd3gfA==", "Sec-WebSocket-Version" : "13"})
        conn = self.create_connection()
        conn.request("GET", url, "", headers)
        print('request: GET %s %s' % (url, '\n'.join(['%s: %s' % (key, headers[key]) for key in headers])))
        resp = conn.getresponse()
        resp.body = resp.read(resp.getheader("content-length"))
        resp.code = resp.status
        resp.headers = self.array_to_dict(resp.getheaders())
        print('reponse: %d %s, ctx: %s' % (resp.status, resp.body, resp.headers["y-context"]))
        return (resp, WebSocket(conn.sock))

    def array_to_dict(self, array):
        result = dict()
        for a in array:
            result[a[0]] = a[1]
        return result

def _load_tvm_ticket(test_app):
    tickets_dir = os.environ['TVM_TICKETS_DIR'] if 'TVM_TICKETS_DIR' in os.environ else '.'
    return open('{}/{}'.format(tickets_dir, test_app)).read().strip()

class XivaApiV2(HTTPRaw):
    """ front port xiva client with raw HTTP & WebSocket interface """
    def __init__(self, **kwargs):
        self.publisher_tst_ticket = _load_tvm_ticket('publisher-tst')
        self.publisher_suspended_ticket = _load_tvm_ticket('publisher-suspended')
        self.publisher_production_ticket = _load_tvm_ticket('publisher-production')
        self.subscriber_tst_ticket = _load_tvm_ticket('subscriber-tst')
        super(self.__class__, self).__init__(**kwargs)


    def prepare_url(self, url, **getparams):
        params = [(key, getparams[key]) for key in getparams if getparams[key] != None]
        quoted_params = urllib.urlencode(params)
        return url + quoted_params

    def auth_headers(self, oauth_token=None, xiva_token=None, tvm_ticket=None):
        headers = {}
        if oauth_token != None:
            headers['Authorization'] = 'OAuth ' + oauth_token
        if xiva_token != None:
            headers['Authorization'] = 'Xiva ' + xiva_token
        if tvm_ticket != None:
            headers['X-Ya-Service-Ticket'] = tvm_ticket
        return headers

    # get-params can be: service, app_name, platform, uuid, push_token, uid, token(xiva-token)
    def subscribe_app(self, oauth_token=None, xiva_token=None, tvm_ticket=None, **getparams):
        url = '/v2/subscribe/app?'
        headers = self.auth_headers(oauth_token, xiva_token, tvm_ticket)
        resp = self.POST(self.prepare_url(url, **getparams), headers)
        return resp

    # get-params can be: service, uid, client, session, token, subscription
    def subscribe_webpush(self, token=None, **getparams):
        url = '/v2/subscribe/webpush?'
        headers = {}
        if token != None:
            headers['Authorization'] = 'Xiva ' + token
        resp = self.POST(self.prepare_url(url, **getparams), headers)
        return resp

    # get-params can be: service, uid, session, token
    def unsubscribe_webpush(self, **getparams):
        url = '/v2/unsubscribe/webpush?'
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # get-params can be: service, uuid, push_token, uid, token(xiva-token)
    def unsubscribe_app(self, oauth_token=None, xiva_token=None, tvm_ticket=None, **getparams):
        url = '/v2/unsubscribe/app?'
        headers = self.auth_headers(oauth_token, xiva_token, tvm_ticket)
        resp = self.POST(self.prepare_url(url, **getparams), headers)
        return resp

    def secret_sign(self, xiva_token=None, tvm_ticket=None, **getparams):
        url = '/v2/secret_sign?'
        headers = self.auth_headers(xiva_token=xiva_token, tvm_ticket=tvm_ticket)
        resp = self.GET(self.prepare_url(url, **getparams), headers)
        return resp

    def verify_secret_sign(self, **getparams):
        url = '/v2/verify_secret_sign?'
        headers = {}
        resp = self.GET(self.prepare_url(url, **getparams), headers)
        return resp

    def watch_subscribers(self, **getparams):
        url = '/v2/watch/subscribers?'
        headers = {}
        return self.WS(self.prepare_url(url, **getparams), headers)

    def subscribe_websocket(self, headers={}, **getparams):
        url = "/v2/subscribe/websocket?"
        return self.WS(self.prepare_url(url, **getparams), headers)

    def subscribe_sse(self, headers={}, **getparams):
        url = "/v2/subscribe/sse?"
        url = self.prepare_url(url, **getparams)
        conn = httplib.HTTPConnection(self.host, self.port)
        conn.request("GET", url, "", headers)
        resp = conn.getresponse()
        resp.headers = resp.getheaders()
        resp.code = resp.status
        resp.headers = self.array_to_dict(resp.getheaders())
        print('reponse: %d, ctx: %s' % (resp.status, resp.headers["y-context"]))
        if resp.code == 200:
            resp.chunked = False
            resp.recv_message = types.MethodType(recv_message,resp)
        else:
            resp.body = resp.read(resp.getheader("content-length"))
        return resp

    def subscriptions(self, **getparams):
        url = '/v2/subscriptions?'
        headers = {}
        resp = self.GET(self.prepare_url(url, **getparams), headers)
        if resp.code == 200:
            resp.body = json.loads(resp.body)
        return resp

    def send(self, body="", xiva_token=None, tvm_ticket=None, **getparams):
        url = '/v2/send?'
        headers = self.auth_headers(xiva_token=xiva_token, tvm_ticket=tvm_ticket)
        return self.POST(self.prepare_url(url, **getparams), headers, body)

    def stream_send(self, body="", xiva_token=None, tvm_ticket=None, **getparams):
        url = '/v2/stream_send?'
        headers = self.auth_headers(xiva_token=xiva_token, tvm_ticket=tvm_ticket)
        return self.POST(self.prepare_url(url, **getparams), headers, body)

    def send_direct(self, body="", **getparams):
        url = '/v2/send?'
        resp = self.POST(self.prepare_url(url, **getparams), {'X-DeliveryMode':'direct'}, body)
        if resp.code == 200:
            resp.body = json.loads(resp.body)
        return resp

    def batch_send(self, body="", **getparams):
        url = '/v2/batch_send?'
        resp = self.POST(self.prepare_url(url, **getparams), {}, body)
        if resp.code == 200:
            resp.body = json.loads(resp.body)
        return resp

    def wild_send(self, body="", **getparams):
        url = '/beta/wild/send?'
        return self.POST(self.prepare_url(url, **getparams), {}, body)


class XivaApiExt(HTTPRaw):
    """ Ext api - mostly passthrough to mobile """
    def __init__(self, **kwargs):
        super(self.__class__, self).__init__(**kwargs)

    def prepare_url(self, url, **getparams):
        params = [(key, getparams[key]) for key in getparams if getparams[key]]
        quoted_params = urllib.urlencode(params)
        return url + quoted_params

    # params can be: app_name, push_token, token, ttl, payload, x-params
    def mobile_push_apns(self, **getparams):
        url = '/ext/v1/mobile_push/apns?'
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # params can be: app_name, push_token, token, ttl, payload, x-params
    def mobile_push_gcm(self, **getparams): # gcm_compatibility
        url = '/ext/v1/mobile_push/gcm?' # gcm_compatibility
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # params can be: app_name, push_token, token, ttl, payload, x-params
    def mobile_push_fcm(self, **getparams):
        url = '/ext/v1/mobile_push/fcm?'
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # params can be: app_name, push_token, token, ttl, payload, x-params
    def mobile_push_mpns(self, **getparams):
        url = '/ext/v1/mobile_push/mpns?'
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # params can be: app_name, push_token, token, ttl, payload, x-params
    def mobile_push_wns(self, **getparams):
        url = '/ext/v1/mobile_push/wns?'
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # params can be: app_name, push_tokens, token, ttl, payload, x-params
    def mobile_batch_push_gcm(self, **getparams): # gcm_compatibility
        url = '/ext/v1/mobile_batch_push/gcm?' # gcm_compatibility
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # params can be: app_name, push_tokens, token, ttl, payload, x-params
    def mobile_batch_push_fcm(self, **getparams):
        url = '/ext/v1/mobile_batch_push/fcm?'
        resp = self.POST(self.prepare_url(url, **getparams), {})
        return resp

    # get-params can be: service, uuid, push_token, uid, token(xiva-token)
    def schedule_unsubscribe_app(self, **getparams):
        url = '/ext/v2/schedule_unsubscribe/app?'
        headers = {}
        resp = self.POST(self.prepare_url(url, **getparams), headers)
        return resp

    def bb_login(self, **getparams):
        url = '/ext/v2/bb/login?'
        headers = {}
        resp = self.POST(self.prepare_url(url, **getparams), headers)
        return resp

    def bb_logout(self, **getparams):
        url = '/ext/v2/bb/logout?'
        headers = {}
        resp = self.POST(self.prepare_url(url, **getparams), headers)
        return resp

class XivaApiBack(HTTPRaw):
    """ back port xiva API client """
    def __init__(self, **kwargs):
        super(self.__class__, self).__init__(**kwargs)

    def prepare_url(self, url, **getparams):
        params = [(key, getparams[key]) for key in getparams if getparams[key]]
        quoted_params = urllib.urlencode(params)
        return url + quoted_params

    def notify(self, service, message, **getparams):
        url = '/notify/' + service + '?'
        body = msgpack.packb(message)
        resp = self.POST(self.prepare_url(url, **getparams), {}, body)
        return resp

class XivaWebPushApi(HTTPRaw):
    """ Web Push API client """
    def __init__(self, **kwargs):
        super(self.__class__, self).__init__(**kwargs)
        self.ws = None

    def connect(self, headers={}):
        resp = self.GET("/v2/vapid_key", {})
        assert_ok(resp)
        self.public_key = resp.body
        (resp, self.ws) = self.WS('/webpushapi/json_rpc', headers)
        eq_(resp.code, 101)
        ws_message = self.ws.recv_message()
        assert_equal(ws_message, None)
        return self.ws

    def make_request(self, method, base_params, extra_params):
        params = base_params.copy()
        for key in extra_params:
            params[key] = extra_params[key]
        return {"method": method, "params": params}

    def check_and_extract_body(self, resp):
        assert_not_equal(resp, None)
        eq_(resp.status_code, 0)
        json_resp = json.loads(resp.payload)
        eq_(json_resp["error"], None)
        return json_resp["result"]

    def subscribe(self, **extra_params):
        base_params = {"public_key": self.public_key, "uuid": "some-instance-id"}
        self.ws.send_message(json.dumps(self.make_request("/webpushapi/subscribe", base_params, extra_params)))
        time.sleep(0.1)
        return self.check_and_extract_body(self.ws.recv_message())

    def unsubscribe(self, **extra_params):
        base_params = {}
        self.ws.send_message(json.dumps(self.make_request("/webpushapi/unsubscribe", base_params, extra_params)))
        time.sleep(0.1)
        return self.check_and_extract_body(self.ws.recv_message())

class XivaClient:
    """ high-level xiva client """
    def __init__(self, host, port, back_port, stat_port=8091):
        self.api = XivaApiV2(host=host, port=port)
        self.api_back = XivaApiBack(host=host, port=back_port)
        self.stat_port = stat_port

    def secret_sign(self, **getparams):
        return json.loads(self.api.secret_sign(**getparams).body)

    def notify(self, service, message):
        return self.api_back.notify(service, message)

    def overview(self):
        return self.api_back.GET("/overview", {}).body

    def stat(self):
        sock = socket.socket()
        sock.connect((self.api.host, self.stat_port))

        ret = ''
        while True:
            part = sock.recv(1005000)
            if not part:
                break
            ret += part
        sock.close()
        return json.loads(ret)

class MeshApi(HTTPRaw):
    def __init__(self, **kwargs):
        super(self.__class__, self).__init__(**kwargs)

    def prepare_url(self, url, **getparams):
        params = [(key, getparams[key]) for key in getparams if str(getparams[key])]
        print params
        quoted_params = urllib.urlencode(params)
        return url + quoted_params

    # get-params can be: service, app_name, platform, uuid, push_token, uid, token(xiva-token)
    def send(self, msg="", **getparams):
        url = '/send?'
        headers = {}
        resp = self.POST(self.prepare_url(url, **getparams), headers, msgpack.packb(msg))
        return resp

class XivaIdmApi(HTTPRaw):
    def __init__(self, **kwargs):
        super(self.__class__, self).__init__(**kwargs)

    def info(self):
        return self.GET('/idm/info', {})

    def get_all_roles(self):
        return self.GET('/idm/get-all-roles', {})

    def make_update_params(self, subject_type, role, login, other_params):
        params = [(key, other_params[key]) for key in other_params if str(other_params[key])]
        if subject_type: params += [('subject_type', subject_type, )]
        if role: params += [('role', role, )]
        if login: params += [('login', login, )]
        return params

    def add_role(self, subject_type=None, role=None, login=None, **other_params):
        params = self.make_update_params(subject_type, role, login, other_params)
        return self.GET('/idm/add-role?' + urllib.urlencode(params), {})

    def remove_role(self, subject_type=None, role=None, login=None, **other_params):
        params = self.make_update_params(subject_type, role, login, other_params)
        return self.GET('/idm/remove-role?' + urllib.urlencode(params), {})

class XivaWebuiApi(HTTPRaw):
    def __init__(self, **kwargs):
        super(self.__class__, self).__init__(**kwargs)

    def parse_response(self, resp):
        resp_json = None
        try:
            resp_json = json.loads(resp.body)
        except:
            pass
        return resp_json

    def list(self, headers):
        resp = self.GET("/webui/list", headers)
        return resp, self.parse_response(resp)

    def service_create(self, headers, params, data):
        resp = self.POST(
            "/webui/service/create?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def service_update(self, headers, params, data):
        resp = self.POST(
            "/webui/service/update?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def service_revoke(self, headers, params, data):
        resp = self.POST(
            "/webui/service/revoke?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def stoken_update(self, headers, params, data):
        resp = self.POST(
            "/webui/send_token/update?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def stoken_revoke(self, headers, params, data):
        resp = self.POST(
            "/webui/send_token/revoke?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def ltoken_update(self, headers, params, data):
        resp = self.POST(
            "/webui/listen_token/update?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def ltoken_revoke(self, headers, params, data):
        resp = self.POST(
            "/webui/listen_token/revoke?" + urllib.urlencode(params), headers, json.dumps(data)
        )
        return resp, self.parse_response(resp)

    def app_info(self, headers, params):
        resp = self.GET('/webui/app/info?' + urllib.urlencode(params), headers)
        return resp, self.parse_response(resp)

    def app_update(self, headers, params, data):
        resp = self.POST('/webui/app/update?' + urllib.urlencode(params), headers, json.dumps(data))
        return resp, self.parse_response(resp)

    def app_revert(self, headers, params, data):
        resp = self.POST('/webui/app/revert?' + urllib.urlencode(params), headers, json.dumps(data))
        return resp, self.parse_response(resp)

    def app_revoke(self, headers, params, data):
        resp = self.POST('/webui/app/revoke?' + urllib.urlencode(params), headers, json.dumps(data))
        return resp, self.parse_response(resp)
