#include "time.h"

namespace yxiva { namespace asn1 {

operation::result convert_asn1time(const ASN1_TIME* time, int64_t& ts)
{
    static const char* FORMAT_STRICT = "%y%m%d%H%M%S%Z";
    static const char* FORMAT_SHORT = "%y%m%d%H%M%Z";
    static const char* FORMAT_STRICT_WITH_OFFSET = "%y%m%d%H%M%S";
    static const char* FORMAT_SHORT_WITH_OFFSET = "%y%m%d%H%M";

    if (!time) return "missing time";
    bool time_with_offset = false;

    // Use ASN1_TIME_to_tm after updating openssl to 1.1.1.
    // Check that time has supported format.
    if (time->type != V_ASN1_UTCTIME) return "unsupported time format";
    const char* format;
    switch (time->length)
    {
    // YYMMDDhhmmZ
    case 11:
        format = FORMAT_SHORT;
        break;
    // YYMMDDhhmmssZ
    case 13:
        format = FORMAT_STRICT;
        break;
    // YYMMDDhhmm+hhmm
    case 15:
        format = FORMAT_SHORT_WITH_OFFSET;
        time_with_offset = true;
        break;
    // YYMMDDhhmmss+hhmm
    case 17:
        format = FORMAT_STRICT_WITH_OFFSET;
        time_with_offset = true;
        break;
    default:
        return "unknown format";
    }

    tm tm{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    auto data = reinterpret_cast<char*>(time->data);
    auto res = strptime(data, format, &tm);
    if (!res || res - data != time->length - (time_with_offset ? 5 : 0))
    {
        return "conversion failed";
    }

    // ASN1_TIME is stored conforming to RFC 5280:
    //   Where YY is greater than or equal to 50, the year SHALL be
    //   interpreted as 19YY; and
    //   Where YY is less than 50, the year SHALL be interpreted as 20YY.
    // While strptime has ranges [69–99] and [00–68] respectively.
    if (data[0] >= '5' && tm.tm_year > 100) tm.tm_year -= 100;

    ts = static_cast<int64_t>(timegm(&tm));

    if (time_with_offset)
    {
        auto tail_pos = res;
        if (tail_pos[0] != '+' && tail_pos[0] != '-') return "unknown format";

        int tail;
        if (!sscanf(tail_pos, "%5d", &tail)) return "conversion failed";

        int64_t diff = (tail % 100) * 60 + (tail / 100) * 3600;
        ts -= diff;
    }

    return ts == -1 ? "conversion error" : operation::success;
}

}}
