#include <yxiva/core/callbacks.h>
#include <yplatform/util/sstream.h>

namespace yxiva { namespace callback_uri {

string mobile_uri(const string& app_name, const string& push_token)
{
    string result;
    result.reserve(SCHEME_MOBILE.size() + app_name.size() + 1 + push_token.size());
    result.append(SCHEME_MOBILE);
    result.append(app_name);
    result.append("/");
    result.append(push_token);
    return result;
}

bool is_mobile_uri(const string& uri)
{
    return (uri.compare(0, SCHEME_MOBILE.size(), SCHEME_MOBILE) == 0);
}

operation::result parse_mobile_uri(const string& uri, string& app_name, string& push_token)
{
    auto pos = uri.find('/', SCHEME_MOBILE.size());
    if (pos != string::npos)
    {
        app_name = uri.substr(SCHEME_MOBILE.size(), pos - SCHEME_MOBILE.size());
        push_token = uri.substr(pos + 1);
        if (!app_name.empty() && !push_token.empty())
        {
            return operation::success;
        }
    }
    return "invalid URI";
}

string webpush_uri(const string& urlencoded_subscription)
{
    string result;
    result.reserve(SCHEME_WEBPUSH.size() + urlencoded_subscription.size());
    result.append(SCHEME_WEBPUSH);
    result.append(urlencoded_subscription);
    return result;
}

bool is_webpush_uri(const string& string)
{
    return (string.compare(0, SCHEME_WEBPUSH.size(), SCHEME_WEBPUSH) == 0);
}

operation::result parse_webpush_uri(const string& uri, string& urlencoded_subscription)
{
    if (uri.size() > SCHEME_WEBPUSH.size())
    {
        urlencoded_subscription = uri.substr(SCHEME_WEBPUSH.size());
        return operation::success;
    }
    return "invalid URI";
}

operation::result parse_webpush_uri_origin(const string& uri, string& origin)
{
    static const string PREF_HTTP = "http://";
    static const string PREF_HTTPS = "https://";
    auto pref_len = uri.find(PREF_HTTPS) == 0 ? PREF_HTTPS.size() :
                                                (uri.find(PREF_HTTP) == 0 ? PREF_HTTP.size() : 0);
    // Don't log details for security reasons.
    if (!pref_len) return "invalid webpush endpoint format";
    origin = pref_len ? uri.substr(0, uri.find("/", pref_len)) : "null";
    return {};
}

operation::result parse_webpush_uri(
    const string& uri,
    string& origin,
    string& urlencoded_subscription)
{
    auto ret = parse_webpush_uri(uri, urlencoded_subscription);
    if (!ret) return ret;
    return parse_webpush_uri_origin(uri, origin);
}

bool is_active_uri(const string& string)
{
    return (string.compare(0, SCHEME_INACTIVE.size(), SCHEME_INACTIVE) != 0);
}

// Return something that makes is_active_uri say 'false'.
string inactive_uri(const string& uri)
{
    auto scheme_end = uri.find(':');
    if (scheme_end == string::npos)
    {
        return SCHEME_INACTIVE + uri;
    }

    return SCHEME_INACTIVE + uri.substr(scheme_end + 1); // skip the ':' in uri
}

string apns_queue_uri(const string& service, const string& app_name, const string& device)
{
    string result;
    yplatform::sstream uri(
        result, SCHEME_APNS_QUEUE.size() + service.size() + app_name.size() + device.size() + 2);
    uri << SCHEME_APNS_QUEUE << service << "/" << app_name << "/" << device;
    return result;
}

bool is_apns_queue_uri(const string& uri)
{
    return uri.compare(0, SCHEME_APNS_QUEUE.size(), SCHEME_APNS_QUEUE) == 0;
}

operation::result parse_apns_queue_uri(
    const string& uri,
    string& service,
    string& app_name,
    string& device)
{
    auto pos1 = uri.find('/', SCHEME_APNS_QUEUE.size());
    auto pos2 = pos1 == string::npos ? string::npos : uri.find('/', pos1 + 1);
    if (pos1 != string::npos && pos2 != string::npos)
    {
        service = uri.substr(SCHEME_APNS_QUEUE.size(), pos1 - SCHEME_APNS_QUEUE.size());
        app_name = uri.substr(pos1 + 1, pos2 - pos1 - 1);
        device = uri.substr(pos2 + 1);
        if (service.size() && app_name.size() && device.size())
        {
            return operation::success;
        }
    }
    return "invalid URI";
}

string xiva_websocket_uri(const string& callback)
{
    return SCHEME_XIVA_WEBSOCKET + callback;
}

bool is_xiva_websocket_uri(const string& uri)
{
    return uri.compare(0, SCHEME_XIVA_WEBSOCKET.size(), SCHEME_XIVA_WEBSOCKET) == 0;
}

std::pair<operation::result, string> parse_xiva_websocket_uri(const string& uri)
{
    if (!is_xiva_websocket_uri(uri)) return { "invalid URI", string{} };
    return { operation::success, uri.substr(SCHEME_XIVA_WEBSOCKET.size()) };
}

}}
