#include <yxiva/core/filter/filter_set.h>
#include <yxiva/core/callbacks.h>
#include <yxiva/core/packing.hpp>
#include <yxiva/core/message.h>
#include <yxiva/core/json.h>
#include <yxiva/core/subscriptions.h>
#include <yxiva/core/platforms.h>
#include <map>

namespace yxiva {

namespace filter {

template <>
bool condition_uses_key(const message_condition_type type)
{
    switch (type)
    {
    case message_condition_type::data_field_equals:
    case message_condition_type::has_least_one:
        return true;
    default:
        return false;
    }
}

template <>
bool condition_uses_key(const subscription_condition_type /*type*/)
{
    return false;
}

template <>
message_condition_type condition_from_string(std::string const& str)
{
    static const std::map<std::string, message_condition_type> CONDITION_TYPE_BY_NAME = {
        { "data_field_equals", message_condition_type::data_field_equals },
        { "$eq", message_condition_type::data_field_equals },
        { "has_tags", message_condition_type::has_tags },
        { "$has_tags", message_condition_type::has_tags },
        { "$event", message_condition_type::event },
        { "$has", message_condition_type::has_least_one }
    };

    auto it = CONDITION_TYPE_BY_NAME.find(str);
    return it == CONDITION_TYPE_BY_NAME.end() ? message_condition_type::unknown : it->second;
}

template <>
subscription_condition_type condition_from_string(std::string const& str)
{
    static const std::map<std::string, subscription_condition_type> CONDITION_TYPE_BY_NAME = {
        { "platform", subscription_condition_type::platform },
        { "transport", subscription_condition_type::transport },
        { "subscription_id", subscription_condition_type::subscription_id },
        { "session", subscription_condition_type::session },
        { "uuid", subscription_condition_type::uuid },
        { "device", subscription_condition_type::device },
        { "app", subscription_condition_type::app }
    };

    auto it = CONDITION_TYPE_BY_NAME.find(str);
    return it == CONDITION_TYPE_BY_NAME.end() ? subscription_condition_type::unknown : it->second;
}

template <>
const char* condition_to_string(const message_condition_type cond)
{
    switch (cond)
    {
    case message_condition_type::data_field_equals:
        return "$eq";
    case message_condition_type::has_tags:
        return "$has_tags";
    case message_condition_type::event:
        return "$event";
    case message_condition_type::has_least_one:
        return "$has";
    default:
        return "unknown";
    }
}

template <>
const char* condition_to_string(const subscription_condition_type cond)
{
    switch (cond)
    {
    case subscription_condition_type::platform:
        return "platform";
    case subscription_condition_type::transport:
        return "transport";
    case subscription_condition_type::subscription_id:
        return "subscription_id";
    case subscription_condition_type::session:
        return "session";
    case subscription_condition_type::uuid:
        return "uuid";
    case subscription_condition_type::device:
        return "device";
    case subscription_condition_type::app:
        return "app";
    default:
        return "unknown";
    }
}

action action_from_string(std::string const& str)
{
    if (str == "send_bright")
    {
        return action::send_bright;
    }
    else if (str == "send_silent")
    {
        return action::send_silent;
    }
    else if (str == "skip")
    {
        return action::skip;
    }
    else
    {
        return action::unknown;
    }
}

const char* action_to_string(const action act)
{
    switch (act)
    {
    case action::send_bright:
        return "send_bright";
    case action::send_silent:
        return "send_silent";
    case action::skip:
        return "skip";
    default:
        return "unknown";
    }
}

bool data_filed_equals_condition(
    const message& message,
    const std::string& key,
    const std::vector<string>& value)
{
    for (auto& val : value)
    {
        auto found = message.data.find(key);
        if (found != message.data.end() && found->second == val)
        {
            return true;
        }
    }
    return false;
}

bool has_tags_condition(const message& message, const std::vector<string>& value)
{
    for (auto& val : value)
    {
        if (!message.has_tag(val))
        {
            return false;
        }
    }
    return true;
}

bool event_condition(const message& message, const std::vector<string>& value)
{
    for (auto& val : value)
    {
        if (message.operation == val)
        {
            return true;
        }
    }
    return false;
}

bool has_least_one_condition(
    const message& message,
    const std::string& key,
    const std::vector<string>& value)
{
    auto found = message.data.find(key);
    if (found == message.data.end()) return false;
    // the array supposed to be sorted and packed on xiva_server when processing /vX/send
    std::vector<string> data_values;
    try
    {
        unpack(found->second, data_values);
    }
    catch (const std::bad_cast& ex)
    {
        return false;
    }
    auto begin = data_values.begin();
    auto end = data_values.end();
    for (auto& val : value)
    {
        if (std::binary_search(begin, end, val)) return true;
    }
    return false;
}

template <>
bool basic_condition<message, message_condition_type>::matches(const message& message) const
{
    switch (type)
    {
    case message_condition_type::data_field_equals:
        return data_filed_equals_condition(message, key, value);
        break;
    case message_condition_type::has_tags:
        return has_tags_condition(message, value);
    case message_condition_type::event:
        return event_condition(message, value);
    case message_condition_type::has_least_one:
        return has_least_one_condition(message, key, value);
    default:
        break;
    }
    return false;
}

bool transport_condition(const sub_t& subscription, const std::vector<string>& value)
{
    auto& cb = subscription.callback_url;
    for (auto& transport : value)
    {
        bool matched = (transport == "mobile" &&
                        (callback_uri::is_mobile_uri(cb) || callback_uri::is_apns_queue_uri(cb))) ||
            (transport == "webpush" && callback_uri::is_webpush_uri(cb)) ||
            (transport == "websocket" && callback_uri::is_xiva_websocket_uri(cb)) ||
            (transport == "http" && !callback_uri::is_mobile_uri(cb) &&
             !callback_uri::is_apns_queue_uri(cb) && !callback_uri::is_webpush_uri(cb));
        if (matched) return true;
    }
    return false;
}

bool field_condition(const string& field, const std::vector<string>& value)
{
    // Value for message-side conditions is expected to be sorted.
    return std::binary_search(value.begin(), value.end(), field);
}

bool app_condition(const sub_t& subscription, const std::vector<string>& value)
{
    auto& url = subscription.callback_url;
    if (callback_uri::is_mobile_uri(url))
    {
        string app;
        string push_token;
        if (callback_uri::parse_mobile_uri(url, app, push_token))
        {
            return field_condition(app, value);
        }
    }
    return false;
}

template <typename ResolveFunction>
bool field_condition_resolved(
    const string& field,
    const std::vector<string>& values,
    ResolveFunction& resolve)
{
    for (auto& value : values)
    {
        if (resolve(value).supported && resolve(value).name == resolve(field).name) return true;
    }
    return false;
}

bool subscription_uuid_condition(const sub_t& subscription, const std::vector<string>& value)
{
    for (auto& uuid : value)
    {
        if (subscription.session_key == canonize_device_id(uuid)) return true;
    }
    return false;
}

template <>
bool basic_condition<sub_t, subscription_condition_type>::matches(const sub_t& subscription) const
{
    switch (type)
    {
    case subscription_condition_type::transport:
        return transport_condition(subscription, value);
    case subscription_condition_type::platform:
        return field_condition_resolved(subscription.platform, value, platform::resolve_alias);
    case subscription_condition_type::subscription_id:
        return field_condition(subscription.id, value);
    case subscription_condition_type::session:
        return field_condition(subscription.session_key, value);
    case subscription_condition_type::device:
        return field_condition(subscription.device, value);
    case subscription_condition_type::uuid:
        return subscription_uuid_condition(subscription, value);
    case subscription_condition_type::app:
        return app_condition(subscription, value);
    default:
        return false;
    }
}

bool expression::operator==(const expression& other) const
{
    return operators == other.operators && operands == other.operands;
}

json_value condition_to_json(const filter::condition& cond);

json_value expression_to_json(const filter::expression& expr, const vars_t& vars)
{
    if (expr.empty()) return json_value(json_type::tstring);

    if (expr.operators.empty())
    {
        auto& var_name = *expr.operands.begin();
        if (filter::is_fake_variable(var_name))
        {
            auto ivar = vars.find(var_name);
            assert(ivar != vars.end());
            return condition_to_json(ivar->second);
        }
    }

    string str;
    str.reserve(expr.operands.size() + expr.operators.size());
    auto ioperand = expr.operands.begin();
    for (auto op : expr.operators)
    {
        switch (op)
        {
        case OP_NOT:
            str.append("!");
            break;
        case OP_AND:
            str.append(*ioperand);
            str.append("&");
            ++ioperand;
            break;
        }
    }
    str.append(*ioperand);

    return str;
}

bool evaluate_operand_value(
    const string& name,
    const message& msg,
    std::map<string, bool>& cached_vars,
    const vars_t& definitions)
{
    auto icached_value = cached_vars.find(name);
    if (icached_value != cached_vars.end()) return icached_value->second;

    bool result = definitions.find(name)->second.matches(msg);
    cached_vars.insert(icached_value, std::make_pair(name, result));
    return result;
}

bool rule::matches(
    const message& msg,
    std::map<std::string, bool>& cached_vars,
    const vars_t& definitions) const
{
    if (expr.empty()) return true;

    bool accumulator = true;
    auto ioperand = expr.operands.begin();
    bool top_operand_value = evaluate_operand_value(*ioperand, msg, cached_vars, definitions);
    for (auto op : expr.operators)
    {
        if (!accumulator) break; // because only AND binary operator supported
        switch (op)
        {
        case OP_NOT:
            top_operand_value = !top_operand_value;
            break;
        case OP_AND:
            accumulator &= top_operand_value;
            ++ioperand;
            top_operand_value = evaluate_operand_value(*ioperand, msg, cached_vars, definitions);
            break;
        }
    }
    accumulator &= top_operand_value;
    return accumulator;
}

bool rule::operator==(const rule& other) const
{
    return action == other.action && expr == other.expr;
}

bool is_fake_variable(const string& var_name)
{
    auto begin = var_name.begin();
    auto end = var_name.end();
    return std::find_if_not(begin, end, isdigit) == end;
}

void add_single_variable_rule(rules_t& rules, vars_t& vars, condition&& cond, action act)
{
    auto var_name = std::to_string(vars.size() + 1);
    assert(vars.count(var_name) == 0);
    vars.emplace(var_name, std::move(cond));
    rules.push_back({ act, { {}, { var_name } } });
}

json_value rule_to_json(const filter::rule& rule, const vars_t& vars = {})
{
    json_value json;
    json["do"] = action_to_string(rule.action);
    if (!rule.expr.empty()) json["if"] = expression_to_json(rule.expr, vars);
    return json;
}

json_value condition_to_json(const filter::condition& cond)
{
    json_value json;
    json_value condition_values(json_type::tarray);
    for (auto&& item : cond.value)
        condition_values.push_back(item);
    if (cond.key.size())
    {
        json[cond.key][condition_to_string(cond.type)] = condition_values;
    }
    else
    {
        json[condition_to_string(cond.type)] = condition_values;
    }
    return json;
}

}

filter_set::filter_set() : default_action_(filter::action::send_bright)
{
}

filter_set::filter_set(
    filter::rules_t&& rules,
    filter::vars_t&& vars,
    filter::action default_action)
    : rules_(std::move(rules)), vars_(std::move(vars)), default_action_(default_action)
{
}

filter::action filter_set::apply(const message& message) const
{
    if (message.has_flag(message_flags::ignore_filters))
    {
        return filter::action::send_bright;
    }
    std::map<std::string, bool> cached_vars;
    for (auto irule = rules_.begin(); irule != rules_.end(); ++irule)
    {
        if (irule->matches(message, cached_vars, vars_))
        {
            return irule->action;
        }
    }
    return default_action_;
}

const std::string& filter_set::to_string()
{
    if (json_str_.empty() && (!rules_.empty() || default_action_ != filter::action::send_bright))
    {
        json_value json;
        auto&& rules = json["rules"];
        rules = json_value(json_type::tarray);
        for (auto const& rule : rules_)
            rules.push_back(filter::rule_to_json(rule, vars_));
        if (default_action_ != filter::action::send_bright)
            rules.push_back(filter::rule_to_json({ default_action_, filter::expression() }));
        auto&& defs = json["vars"];
        defs.set_object();
        for (auto const& var : vars_)
        {
            if (filter::is_fake_variable(var.first)) continue;
            defs[var.first] = condition_to_json(var.second);
        }
        json_str_ = json.stringify();
    }
    return json_str_;
}

bool filter_set::operator==(const filter_set& other) const
{
    return rules_ == other.rules_ && vars_ == other.vars_ &&
        default_action_ == other.default_action_;
}

}
