#include <yxiva/core/filter/parse.h>
#include "parser/decode_filter_v1.h"
#include "parser/decode_filter_v2.h"
#include "parser/definition_translator.h"
#include <yxiva/core/message.h>
#include <yxiva/core/subscriptions.h>
#include <cctype>

namespace yxiva { namespace filter {

operation::result parse(filter_set& dst, const std::string& raw_src, const format_version version)
{
    if (raw_src.empty())
    {
        dst = filter_set();
        return operation::success;
    }

    json_value doc;
    auto res = json_parse(doc, raw_src);
    if (!res) return operation::result(std::string("json parse error: ") + res.error_reason);

    if (version == VERSION2 || doc.has_member("vars"))
    {
        return parser::decode_filter_v2(dst, doc);
    }
    else
    {
        return parser::decode_filter_v1(dst, doc);
    }
}

template <typename Condition>
operation::result decode_condition(Condition& cond, const json_value& json_cond)
{
    parser::basic_definition_translator<Condition> translate;
    if (!translate(json_cond, "")) return translate.get_error();

    translate.move_result_to(cond);
    // Sort values to ensure matches() works as intended.
    std::sort(cond.value.begin(), cond.value.end());

    return operation::success;
}

template operation::result decode_condition<condition>(
    condition& cond,
    const json_value& json_cond);
template operation::result decode_condition<subscription_condition>(
    subscription_condition& cond,
    const json_value& json_cond);

filter::action parse_and_apply(const sub_t& subscription, const message& message)
{
    for (auto& condition : message.subscription_matchers)
    {
        if (!condition.matches(subscription)) return filter::action::skip;
    }
    filter_set filter_set;
    return filter::parse(filter_set, subscription.filter) ? filter_set.apply(message) :
                                                            filter::action::send_bright;
}

operation::result parse_and_apply(
    filter::action& action,
    const sub_t& subscription,
    const message& message)
{
    action = filter::action::skip;
    for (auto& condition : message.subscription_matchers)
    {
        if (!condition.matches(subscription)) return operation::success;
    }
    action = filter::action::send_bright;
    filter_set filter_set;
    if (auto result = filter::parse(filter_set, subscription.filter))
    {
        action = filter_set.apply(message);
        return operation::success;
    }
    else
    {
        return result;
    }
}

}}
